	page	,132

;-----------------------------------------------------------------------------
;
;  This file is part of doskey.com.
; 
;  Copyright (C) 2001-2011 Paul Houle (http://paulhoule.com)
; 
;  This program is free software; you can redistribute it and/or modify
;  it under the terms of the GNU General Public License as published by
;  the Free Software Foundation; either version 2 of the License, or
;  (at your option) any later version.
; 
;  This program is distributed in the hope that it will be useful,
;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;  GNU General Public License for more details.
; 
;  You should have received a copy of the GNU General Public License
;  along with this program; if not, write to the Free Software
;  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
;
;  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;
; Execution proceeds here from command line.  All code and data within is
; temporary (discarded if we install as a TSR) - the area is overlaid by
; the new TSR's BUFSIZE block (used to store the command history and macros).
;
;-----------------------------------------------------------------------------

	.nolist
	include	dkdefs.inc	;include constant definitions, externdef's
	.list
	option	noljmp		;disallow automatic jmp-lengthening

;-----------------------------------------------------------------------------
;
; Jump directly here (to "Startup" label) from DOS entry.
;
; This is a .com image, so all segment registers are the same on entry.
;
; Note: the init code must not alter any run-time variables (variables that
; overlay the PSP), since that would destroy PSP fields that are needed
; during init (e.g., the incoming command line).

Startup:

	push	es			;090629 *** START
	xor	di,di			;si:di= Windows API entry address
	mov	es,di			;  (0:0 if Win 3.1/9x/me not running)
	mov	ax,1602h
	int	2fh
	mov	si,es
	pop	es			;090629 *** END

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	;090612 Kludge to disable QueryClose (Win '95-only) call if running
	;090612 on an OS version that doesn't support the int 2fh function.
	;
	;090629 Added logic to also disable QueryClose if Windows GUI is not
	;090629 running.

	mov ah,30h			;090612 *** START
	int 21h				;get OS version
	or	di,si			;zero if Windows API not present
	.if zero? || al < 7		; if not at least Win '95 & GUI,
	  mov di,offset QueryClose	;cause QueryClose to return non-zero
	  mov ax,0ff0ch			;  write "or al,0ffh" opcode
	  stosw
	  mov al,0c3h			;  write "retn" opcode
	  stosb
	.endif				;090612 *** END


	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Store current int 2fh contents.  This is needed for chaining (in
	; case we go resident), and for uninstalling.

	push	es
	mov	ax,352fh		;save current int 2fh vector
	int	21h
	mov	word ptr Old2fVector,bx
	mov	word ptr Old2fVector+2,es
	pop	es

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Parse CR-terminated command line, and execute appropriate tasks.
	; Note the first thing we do is convert the terminating CR (or 0, if
	; we find that first) into a zero.  This insures we'll NEVER have a
	; zero in the input line, which could be fatal in places.

	mov	si,PSP.pspCommandTail + 1 ;si= addr of incoming command line
	push	si
	.repeat
	  lodsb				;scan to CR or 00
	  .break .if al == 0
	  xor	al,CR
	.until	zero?
	mov	[si-1],al		;terminate/re-terminate with a zero
	pop	si

	.repeat				;start dummy block (to support breaks)

	  ifdef	EBUG
		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; If debug build, issue a warning message.
		;
		; Also, check that the assembly went OK.  This should have
		; been done at assemble time, but MASM 6.11 screws up when
		; evaluating differences between labels on the first "pass."
		; So we do the check at run time during debug builds only.
		;
		; Lastly, shrink our memory block paras to 1 max segment's
		; worth.  This eases debugging a bit by allowing a command
		; shell to be entered during a debug session.

	    push si
	    mov	 si,offset WarDebMsg
	    call DisplayFormattedMsg
	    pop  si
	    mov	cx,offset InstallStackBase ;cx + install stack (256 bytes)
	    sub	cx,offset EndResident	   ;  must fit
	    mov	di,offset GoTSRTooBigMsg ;assume GoTSR overlay region overflow
	    mov	ax,GF_EXIT or GF_ERROR
	    .break .if cx > (BUFSIZE_MIN - MACROS_MAX) ;break= overflow, abort
	    mov	bx,sp
	    add bx,15
	    rcr bx,1
	    mov cl,3
	    shr bx,cl
	    mov ah,4ah
	    int 21h
	  endif

	  call	ParseOpts		;parse any /options
	  .break .if al & GF_EXIT	;break= exit immediately w/message
	  call	AdjustBufsize		;condition final BUFSIZE value
	  call	ParseMacroDef		;parse macro definiton
	  .break .if al & GF_EXIT	;break= exit immediately w/message
	  .if	!(al & GF_REINSTALL)	;if not forcing reinstall,
	    call ModifyTSR		;attempt to modify an installed TSR
	  .endif
	  .if	al & GF_REINSTALL	;if reinstalling or nothing to modify,
	    and	al,TF_RESIDENTFLAGS	;init the resident flags
	    mov	tflags,al
	    mov	si,offset DoskeyInstalledMsg ;issue installation message now
	    call DisplayFormattedMsg
	    call GetMacroDefinitionInfo	;get macro info for InstallTSR
	    jmp	InstallTSR		;jmp= install-new/reinstall TSR
	  .endif
	.until	1			;end dummy block

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Exit points.  ax= flags, di= potential exit message address.

	.if	al & GF_EXIT		;if exit forced w/message,
	  push	ax			;save flags
	  mov	si,di			;display exit message
	  call	DisplayFormattedMsg
	  pop	ax			;restore flags
	.endif
	and	al,GF_ERROR		;set non-zero errolevel, if needed
	mov	ah,4ch			;exit to OS
	int	21h

;-----------------------------------------------------------------------------
; Attempt to modify an installed TSR (caller did not specify /REINSTALL).
; Note: if no version of DOSKEY (compatible or otherwise), is currently
; installed, we force /REINSTALL and exit immediately.
;
; IN:
;   ax= parsed option flags (GF_REINSTALL bit must be clear)
;   SwiData1.iisSize= parsed BUFSIZE
; OUT:
;   ax= parsed option flags: GF_REINSTALL turned on if no TSR installed;
;	otherwise, GF_EXIT/GF_ERROR may be set if an error occurs.
;   di= exit message, if ax & GF_EXIT

ModifyTSR proc	near
	push	es			;save local es
	push	ax			;save parsed option flags

	.repeat				;start dummy block (to support breaks)
	  mov	ax,4800h		;perform DOSKEY installation check
	  int	2fh
	  mov	bl,GF_REINSTALL		;assume we'll force a reinstall
	  .break .if ax == 4800h	;break= not installed, force reinstall
	  mov	bl,GF_EXIT or GF_ERROR	;now assume incompatible TSR error
	  mov	di,offset IncompatDoskeyMsg
	  .break .if ax != DOSKEY_ID	;break= incompatible TSR

		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; Compatible TSR already installed (es= installed TSR
		; segment).  Attempt to modify the installed TSR segment.
		;
		; Only allow modification if user hasn't attempted to
		; specify a different BUFSIZE.

	  pop	ax			;restore flags image
	  push	ax
	  .if	ax & GF_BUFSIZE		;if user specified /BUFSIZE,
	    mov	di,offset CantChangeBufsizeMsg	;assume BUFSIZE change error
	    mov	bl,GF_EXIT or GF_ERROR
	    mov	si,offset SwiData1.iisSize	;test assumption
	    mov	di,si
	    cmpsw
	    .break .if !zero?		;break= assumption correct, abort
	  .endif

		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; Change any resident flags requested by user.

	  and	ax,TF_RESIDENTFLAGS or (TF_RESIDENTFLAGS shl 8)
	  not	ah
	  and	ah,es:tflags
	  or	al,ah
	  mov	es:tflags,al

		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; Process macro definition/redefinition/deletion.

	  call	GetMacroDefinitionInfo	;get macro definition info
	  .if	!zero?			;if a definition/deletion present,
	    call DeleteMacro		;delete if matching entry found
	  .endif
	  jcxz	@F			;jmp= no definition, or deletion only
	    call StoreMacro		;store macro definition
	    .break .if bl != 0		;break= macro storage error
	  @@:

	  pop	ax			;restore flags image
	  push	ax

	  ifdef	EBUG
		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; If requested, simulate an int 2fh get input line call,
		; using this code (so we can debug), but using the data
		; segment of the installed TSR.

	    .if	ax & GF_GETINPUT
		push ax			;save flags
		call GetInput		;set up and simulate int 2f get input
		pop  ax			;restore flags
	    .endif
	  endif

		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; Display installed history and/or macros, if requested.

	  push	ds
	  push	es
	  pop	ds
	  .if	al & GF_HISTORY		;if history display requested,
	    push ax			;save option flags
	    mov  si,slHistory.Head	;display installed command history
	    call ListDisp
	    pop  ax			;restore option flags
	  .endif
	  .if	al & GF_MACROS		;if macros display requested,
	    mov  si,slMacros.Head
	    call ListDisp
	  .endif
	  pop	ds

	  xor	bx,bx			;indicate TSR work successful
	.until	1			;end dummy block

	pop	ax			;restore parsed option flags
	pop	es			;restore local es
	or	al,bl			;set any flags selected above

	ret
ModifyTSR endp


	.repeat
	  call DisplayFormattedMsg	;display next formatted string
	  call OutputCRLF		;follow with a CRLF
ListDisp label near
	.until byte ptr [si] == 0
	retn

ifdef	EBUG
;-----------------------------------------------------------------------------
; Simulate an int 2fh get input line call, using this code (so we can debug),
; but using the data segment of the installed TSR.

GetInput proc near
	push	es			;move in test command line template
	push	ds
	pop	es
	mov	di,offset CmdBufLen
	mov	si,offset TestCmdTemplate
	mov	cx,(TestCmdTemplateEnd - TestCmdTemplate) + 1
	rep movsb
	pop	es

	mov	dx,offset CmdBufLen	;ds:dx= incoming template
	mov	bx,'de'			;set special DEBUG signature
	mov	cx,'ug'
	mov	ax,4810h		;int 2fh DOSKEY get command line
	pushf				;create iret frame
	push	cs
	call	near ptr MyInt2f	;go perform simulation

	mov	si,offset CmdBufLen + 1	;si= returned input line length
	lodsb				;ax= command length, w/o CR
	mov	ah,0
	add	si,ax			;point to CR
	mov	word ptr [si],']'	;terminate with ']',0
	mov	si,offset CmdBufLen + 2	;display entered command line
	push	es
	push	ds
	pop	es
	call	DisplayFormattedMsg
	pop	es

	ret
GetInput endp
endif

;-----------------------------------------------------------------------------
; Delete installed TSR macro, if a matching one exists.  This should only
; be called if a 'macro=xxx' (xxx may be empty) definition was found on the
; command line.
;
; We only delete the stored macro if there will be enough room
; to write the new definition in its place (assuming a "new
; definition" exists, which is indicated by a non-zero cx).
;
; We attempt to keep the Active pointer correct; but, if the active
; entry is deleted, Active is pointed to the string list terminator.
;
; IN:
;   si/cx= definition start/length (as set byGetMacroDefinitionInfo)
;   di= beyond macro '=' separator (as set byGetMacroDefinitionInfo)
;   es= installed TSR segment
; OUT:
;   si/cx= unchanged
;   di= destroyed
;   bx= amount of potential macro freespace following deletion

DeleteMacro proc near
	push	si			;save macro definition info
	push	cx

	mov	bx,es:slHistory.Next	;bx= history freespace that could
	sub	bx,es:slHistory.Head	;  be used to define macro)
	sub	bx,HISTORY_MIN		;(reserve minimum history space)
	.if	carry?			;(should never be below HISTORY_MIN,
	  xor	bx,bx			;but we are, no history space to use
	.endif

	push	cx			;save macro definition length
	push	word ptr [di]		;save definition area we'll modify
	push	di			;save so we undo the modification
	mov	byte ptr [di],0		;terminate definition following '='
	mov	di,es:slMacros.Head	;es:di= TSR macro string list pool
	call	StringListPoolSearch	;search for macro name in string pool
	pop	di			;undo modification of macro definition
	pop	word ptr [di]
	pop	cx			;restore macro definition length
	.if	zero?			;if macro found, delete definition,
	  push	ds			;set up to modify installed TSR memory
	  push	es
	  pop	ds
	  mov	di,si			;scan past end of entry to delete
	  xor	ax,ax
	  .repeat
	    inc	bx			;(update potential freespace)
	    scasb
	  .until zero?
	  .if	cx <= bx		;only if redefined macro would fit,
	    mov	ax,slMacros.Active	;prepare to maintain Active
	    .if	ax < di			;if deletion affects Active,
	      add ax,di			;assume Active entry not deleted
	      sub ax,si
	      .if ax >= di		;if assumption wrong,
		mov ax,slMacros.Tail	;reset to string list terminator
	      .endif
	      mov slMacros.Active,ax	;store modified active pointer
	    .endif
	    mov cx,si			;cx= bytes to pack to higher memory
	    sub cx,slMacros.Head
	    std				;move reverse
	    cmpsb			;point at first byte to move
	    rep movsb			;pack previous data to higher memory
	    cld
	    inc di			;di= point to last byte moved
	    mov slMacros.Head,di		;shrink macro string list pool
	    mov slHistory.Next,di	;enlarge history string list pool
	  .endif
	  pop	ds
	.endif

	pop	cx			;restore macro definition info
	pop	si
	ret
DeleteMacro endp

;-----------------------------------------------------------------------------
; Store installed TSR macro.  This should only be called if a 'macro=xxx'
; (xxx must NOT be empty) definition was found on the command line.
;
; We only store the macro if enough room to write the definition
; is available; if not, we issue an error.
;
; The Active pointers are maintained for both the macro and history
; string lists (the macro pointer will never be lost; the history pointer
; can be if its active entry is deleted).
;
; IN:
;   si/cx= definition start/length (as set byGetMacroDefinitionInfo)
;   es= installed TSR segment
;   bx= amount of macro freespace (set by DeleteMacro).
; OUT:
;   si/cx= unchanged
;   bl= 0 if storage successful, else GF_EXIT or GF_ERROR;
;	if bl non-zero, di holds error message address.

StoreMacro proc near

	cmp	bx,cx			;carry if not enough macro room
	mov	di,offset MacrosFullMsg ;assume macros full error
	mov	bl,GF_EXIT or GF_ERROR
	.if	!carry?			;if room to store macro,
	  push	ds			;set up to modify installed TSR memory
	  push	es
	  pop	ds
	  assume bx:near ptr STRINGLIST
	  call	GetHistoryFreespace	;get history string list freespace

	  sub	[bx].Next,cx		;take freespace from history
	  mov	bx,offset slMacros	;enlarge macro area
	  sub	[bx].Head,cx
	  sub	[bx].Tail,cx
	  sub	[bx].Active,cx
	  mov	di,[bx].Head
	  mov	si,di
	  add	si,cx
	  mov	cx,[bx].Tail
	  inc	cx			;(move terminator too)
	  sub	cx,[bx].Head
	  rep movsb

	  pop	ds			;store macro definition

	  call	GetMacroDefinitionInfo	;get macro definition info
	  call	StringListAppend	;store macro definition

	  xor	bx,bx			;indicate macro storage successful
	.endif

	ret
StoreMacro endp

;-----------------------------------------------------------------------------
; Gets command-line macro definition information.
;
; IN:
;   ds= local segment
;   es= unused
; OUT:
;   si= addr of start of parsed macro definition string (if any)
;   di= addr of character beyond '=' within macro definition string
;   cx= entire macro definition size, if a macro is being defined;
;	zero if no macro definition or macro deletion only ("macro=")
;   non-zero status if macro definition (or deletion) was parsed

GetMacroDefinitionInfo proc near

	mov	si,offset CmdBuf	;si= start of parsed macro definition
	xor	cx,cx
	.if	[si] != cl		;if macro alteration parsed,
	  push	si			;find '=' in macro definition
	  .repeat
	    lodsb
	  .until al == '='
	  mov	di,si			;save address immediately beyond '='
	  pop	si
	  .if [di] != cl		;if not simply deleting macro,
	    mov cl,CmdBufLen		;cx= macro definition length
	    inc cx
	  .endif
	  or	di,di			;set non-zero (flag definition parsed)
	.endif

	ret
GetMacroDefinitionInfo endp

;-----------------------------------------------------------------------------
; Parse command line options, if any.
;
; IN:
;   si= start of 0-terminated command line.
; OUT:
;   ax= parsed option flags
;   di= exit message, if ax & GF_EXIT

ParseOpts proc	near
	xor	ax,ax			;various option flags accumulated here
	push	ax
	.repeat				;loop to parse all options,

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Scan option table for match of option string.

	  call	SkipWhitespace		;skip any leading whitespace
	  .break .if al != '-' && al != '/' ;break= no more options found
	  inc	si			;bypass option indicator

	  push	si			;save option start
	  .repeat			;scan past all option name text
	    lodsb
	    .if	al != '?'		;allow '?' in option name
	      call IsMacroNameChar	;also allow macro name characters
	    .endif
	  .until !zero?
	  dec	si			;point to terminator
	  pop	di			;restore option start
	  push	[si]			;save terminating text
	  push	si			;save address of terminator
	  mov	byte ptr [si],0		;make option asciiz string

	  mov	si,offset OptionTableList ;start of option table
	  .while 1			;loop to scan option table,
	    lodsw			;bx= option handler address
	    xchg bx,ax
	    push si			;save current option table address
	    push di			;save option text address
	    .if  byte ptr [si]		;if haven't hit end of option table,
	      call StringListPoolSearch	;check for option match
	    .endif
	    pop  di			;restore option text address
	    pop	 si			;restore current option table address
	    .break .if zero?		;break= match, or end of table
	    .repeat			;scan to start of next option
	      lodsb
	    .until al == 0
	  .endw

	  pop	si			;si= terminator address
	  pop	word ptr [si]		;restore terminator characters

	  pop	ax			;put flags in ax for option handler
	  call	bx			;execute option handler
	  test	al,GF_EXIT		;non-zero if exit was forced
	  push	ax
	.until	!zero?			;loop if exit was not forced
	pop	ax			;ax= parsed option flags
	ret
ParseOpts endp

;-----------------------------------------------------------------------------

GiveHelp proc	near
	mov	di,offset HelpMsg	;address of message to display
	mov	al,GF_EXIT		;force immediate exit
	ret
GiveHelp endp

;-----------------------------------------------------------------------------

SetInsert proc near
	or	ax,TF_STARTINSERT_CHANGE or TF_STARTINSERT
	ret
SetInsert endp

SetOverstrike proc near
	and	al,not TF_STARTINSERT
	or	ax,TF_STARTINSERT_CHANGE
	ret
SetOverstrike endp

;-----------------------------------------------------------------------------

SetReinstall proc near
	or	al,GF_REINSTALL
	ret
SetReinstall endp

;-----------------------------------------------------------------------------

SetMacros proc near
	or	al,GF_MACROS
	ret
SetMacros endp

;-----------------------------------------------------------------------------

SetHistory proc near
	or	al,GF_HISTORY
	ret
SetHistory endp

;-----------------------------------------------------------------------------
;
; Uninstall logic.  If we recognize an installed DOSKEY, and the int 2f
; vector still points to it, we allow the uninstall; otherwise, immediate
; exit w/error message.

Uninstall proc	near

	push	es
	.repeat			;start dummy block (to support breaks)
	  mov	ax,4800h		;check if DOSKEY installed; if so,
	  int	2fh			;  es= installed segment address
	  cmp	ax,DOSKEY_ID		;zero status if signature correct
	  mov	di,offset CantUninstallMsg	;assume "can't uninstall"
	  mov	ax,GF_EXIT or GF_ERROR		;  message w/errorlevel 1
	  mov	cx,es			;break= int 2fh vector has changed
	  .break .if word ptr Old2fVector + 2 != cx
	  .break .if word ptr Old2fVector != offset MyInt2f

	  push	ds			;restore installed DOSKEY old int 2fh
	  lds	dx,es:Old2fVector	;  vector
	  mov	ax,252fh
	  int	21h
	  pop	ds

	  mov	ah,49h			;release installed memory block
	  int	21h

	  mov	di,offset DoskeyUninstalledMsg	;issue uninstall message
	  mov	ax,GF_EXIT			;  and exit w/0 errorlevel
	.until	1		;end dummy block
	pop	es
	ret

Uninstall endp

;-----------------------------------------------------------------------------

BufSize proc near
	push	ax		;save flags

	call	SkipWhitespace	;skip any whitespace following /BUFSIZE
	cmp	al,'='		;set zero status if option OK so-far
	stc			;(set carry in case it isn't)
	.if	zero?		;if option formed OK so-far,
	  inc	si		;bypass '='
	  call	SkipWhitespace
	  xor	cx,cx		;accumulate number here
	  .repeat		;loop to parse numbers
	    lodsb		;ax= next character, in numeric form
	    cbw			;(zero ah)
	    sub	al,'9' + 1	;set carry if next character is a digit
	    add	al,10
	    .break .if !carry?	;break= no more digits, done w/number
	    xchg ax,cx		;multiply accumulator by 10
	    mov	dx,10
	    mul	dx
	    .break .if carry?	;break= 16-bit overflow
	    add	cx,ax		;cx= updated accumulator
	  .until carry?		;continue if no 16-bit overflow
	  dec	si		;back up in front of terminating character
	  mov	SwiData1.iisSize,cx ;save parsed buffer size, good or bad
	.endif

	pop	ax		;restore flags
	.if	carry?		;if error parsing BUFSIZE,
	  mov	di,offset BadBufSizeSettingMsg	;cause exit with message
	  mov	al,GF_EXIT or GF_ERROR		;  and non-zero errorlevel
	.endif
	or	ax,GF_BUFSIZE	;show BUFSIZE was parsed

	ret
BufSize endp

;-----------------------------------------------------------------------------

ifdef	EBUG
SetGetInput proc near
	or	ax,GF_GETINPUT
	ret
SetGetInput endp
endif

;-----------------------------------------------------------------------------

InvalidOption proc	near
	mov	word ptr [si],CR		;terminate option w/CR,0

	mov	si,offset InvalidOptionMsg	;issue invalid option message
	call	DisplayFormattedMsg

	dec	di				;cause exit with message
	mov	al,GF_EXIT or GF_ERROR		;  and non-zero errorlevel
	ret
InvalidOption endp

;-----------------------------------------------------------------------------
; Adjust BUFSIZE.  We limit the minimum to one worst-case history and
; macro row, and the maximum to the amount of space in our installed
; segment.  Also, we round the final value up to use a multiple of
; paragraphs, so we don't waste a few bytes of space at the end of the
; segment
;
; IN:
;   ax= parsed option flags
;   si= current command line position
;   SwiData1.iisSize= entry BUFSIZE
; OUT:
;   ax= parsed option flags (no change)
;   si= current command line position (no change)
;   SwiData1.iisSize= adjusted BUFSIZE

AdjustBufsize proc near

	push	ax			;save parsed option flags

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Limit /BUFSIZE to a minimum of BUFSIZE_MIN,
	; and a maximum of the amount of memory in the entry .com segment.

	mov	ax,SwiData1.iisSize
	.if	ax < BUFSIZE_MIN	;limit BUFSIZE minimum
	  mov	ax,BUFSIZE_MIN
	.endif
	mov	dx,sp			;limit BUFSIZE maximum
	sub	dx,offset EndResident
	.if	ax > dx
	  xchg	ax,dx
	.endif

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Whatever the BUFSIZE is now, round it up so we use all of the
	; final paragraph in the segment.

	add	ax,offset EndResident + 0fh ;round to next paragraph boundary
	sbb	dx,dx			;limit to 1 pargraph under 64k
	or	ax,dx
	and	al,not 0fh
	sub	ax,offset EndResident
	mov	SwiData1.iisSize,ax

	pop	ax			;restore parsed option flags

	ret
AdjustBufsize endp

;-----------------------------------------------------------------------------
; Parse macro definition.
;
; IN:
;   ax= parsed option flags
;   si= current 0-terminated command line position within CmdBuf
; OUT:
;   ax= parsed option flags, (GF_EXIT or GF_ERROR) set if error.
;   si= advanced beyond parsed, formatted macro definition
;   di= (if error) error message text.

ParseMacroDef proc near

	push	ax			;save parsed option flags

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Parse and format macro definition string (if one supplied on
	; command line).  The formatted macro string is placed at the
	; start of CmdBuf.
	;
	; The Macro name is converted to lowercase, whitespace is removed on
	; either side of the '=' separating the name from the definition,
	; and the string is zero-terminated.  Note whitespace at the end of
	; the definition is retained.
	; The macro length  (w/o the trailing 0) is stored in CmdBufLen.

	.repeat				;start dummy block (to support breaks)
	  call	SkipWhitespace		;skip whitespace before macro name
	  mov	di,offset CmdBuf	;put formatted macro definition here
	  .repeat			;loop to transfer macro name
	    lodsb			;get next suspected name char
	    call ToLower		;convert to lower case
	    stosb			;  and store
	    call IsMacroNameChar	;zero if al is a macro name character
	  .until !zero?			;loop if we stored a valid name char
	  dec	si			;point to macro name terminator
	  dec	di
	  mov	cx,di			;cx= macro name characters parsed
	  sub	cx,offset CmdBuf
	  .break .if zero?		;break= no name found, al= terminator
	  call	SkipWhitespace		;skip whitespace before '='
	  xor	al,'='			;see if next character is '='
	  .break .if !zero?		;break= macro definition error
	  movsb				;move the '='
	  call	SkipWhitespace		;skip any whitespace following '='
	  .repeat			;move text following '='
	    inc	cx			;increase macro length
	    lodsb
	    stosb
	  .until al == 0		;up to terminator
	.until	1			;end dummy block
	mov	CmdBufLen,cl		;store macro size (w/o terminator)

	cmp	al,0			;non-zero if error parsing definition
	pop	ax			;restore parsed option flags
	.if	!zero?			;if definition error,
	  mov	di,offset InvalidMacroDefMsg	;cause exit with message
	  mov	al,GF_EXIT or GF_ERROR		;  and non-zero errorlevel
	.endif

	ret
ParseMacroDef endp

;-----------------------------------------------------------------------------
; Table used to parse command-line options.

DECLARE_OPTION	macro	string,handler
	dw	offset handler
	db	string,0
	endm

OptionTableList label	near	;marks start of option string list
	DECLARE_OPTION	'?',GiveHelp
	DECLARE_OPTION	'i',SetInsert
	DECLARE_OPTION	'o',SetOverstrike
	DECLARE_OPTION	'r',SetReinstall
	DECLARE_OPTION	'h',SetHistory
	DECLARE_OPTION	'm',SetMacros
	DECLARE_OPTION	'u',Uninstall
	DECLARE_OPTION	'b',BufSize
	ifdef	EBUG
	  DECLARE_OPTION 'g',SetGetInput
	endif
	dw	InvalidOption		;terminate option string list
	db	0

;-----------------------------------------------------------------------------
;
; Message strings.  Each message must be terminated with a zero.

ifdef	EBUG
  TestCmdTemplate db IOBUF_SIZE
	db	TestCmdTemplateEnd - TestCmdTemplateStart
  TestCmdTemplateStart db 'Test input line'
  TestCmdTemplateEnd   db 0dh
  GoTSRTooBigMsg db '*** Assemble-time error: overlay region too big ***',CR,0
  WarDebMsg   db '*** Warning: debug build ***',CR,0
endif

CantChangeBufsizeMsg db 'Can''t change BUFSIZE of installed DOSKEY.',CR,0

DoskeyInstalledMsg db 'DOSKEY installed.',CR,0

InvalidMacroDefMsg db 'Invalid macro definition.',CR,0

InvalidOptionMsg db 'Invalid option: ',0

MacrosFullMsg label byte
;030121 db 'Insufficent installed memory to store macro.  Uninstall DOSKEY,',CR
 db 'Insufficent memory for macro.  Uninstall DOSKEY,',CR		;030121
 db 'then reinstall with a larger /BUFSIZE.',CR,0

BadBufSizeSettingMsg db 'Invalid /BUFSIZE=size specification.',CR,0

CantUninstallMsg label byte
 db 'Can''t Uninstall: DOSKEY not resident, vectors have changed, or',CR
IncompatDoskeyMsg label byte	;(share message text)
 db 'An incompatible DOSKEY is installed.',CR,0

DoskeyUninstalledMsg db 'DOSKEY uninstalled.',CR,0

HelpMsg label near
			;030121 Version changed to 1.6, copyright to 2003
			;030926 Version changed to 1.7, added paulhoule.com
			;031022 Version changed to 1.8
			;090612 Version changed to 1.9, copyright to 2009
			;090629 Version changed to 2.0
 db CR
 db 'Enhanced DOSKEY Ver 2.0 - edits, recalls, auto-completes commands.',CR
 db 'Copyright 2009 Paul Houle (paulhoule.com).  All rights reserved.',CR,CR
 db 'DOSKEY [-option [-option]] [macro=[text]]',CR,CR
 db '  -?               This message',CR
 db '  -R{einstall}     Install new instance of DOSKEY',CR
 db '  -U{ninstall}     Remove DOSKEY from memory',CR
 db '  -B{ufsize}=size  Set command recall/macro size (default 512)',CR
 db '  -M{acros}        Display macros',CR
 db '  -H{istory}       Display command recall buffer',CR
 db '  -I{nsert}        Keystroke insert mode',CR
 db '  -O{verstrike}    Keystroke overwrite mode (default)',CR
ifdef	EBUG
 db '  -G{etinput}  *** DEBUG: Simulate int 2fh get input line call',CR
 db '               ***        Uses already-installed TSR segment',CR
endif
 db '  macro=           Name of macro to modify or delete',CR
 db '  [text]           Commands to record (empty to delete macro)',CR,CR
 db 'ESC clears command; UP/DOWN selects history; F7 displays history;',CR
 db 'ALT+F7 clears history; F8/ALT+F8 selects history; F9 selects',CR
 db 'command by number; F10 displays macros; ALT+F10 clears macros;',CR
 db 'TAB/SHIFT+TAB auto-completes partial command or file name;',CR
 db 'ALT+F6 generate tab.',CR,CR
 db 'The following codes are allowed in macros:',CR
;090612  db '$T           Separates commands within a single macro.',CR
 db '$T           Separates commands within a macro.',CR
 db '$L,$G,$B,$$  Replaced with <, >, |, $ when macro executed.',CR
 db '$1-$9        Parameter replacement (similar to batch file %1-%9).',CR
 db '$*           Replaced with entire line following macro name.',CR,0

;-----------------------------------------------------------------------------
;
; End Module
;
;-----------------------------------------------------------------------------

	ENDSEG
	end
