	page	,132

;-----------------------------------------------------------------------------
;
;  This file is part of doskey.com.
; 
;  Copyright (C) 2001-2011 Paul Houle (http://paulhoule.com)
; 
;  This program is free software; you can redistribute it and/or modify
;  it under the terms of the GNU General Public License as published by
;  the Free Software Foundation; either version 2 of the License, or
;  (at your option) any later version.
; 
;  This program is distributed in the hope that it will be useful,
;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;  GNU General Public License for more details.
; 
;  You should have received a copy of the GNU General Public License
;  along with this program; if not, write to the Free Software
;  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
;
;  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;
;		Enhanced DOSKEY
;
; Source for enhanced DOSKEY.
;
; Change history (just started 12/12/2001).
;  20011212:	caused completions to occur in lowercase.
;		fixed bug with forward arrow going beyond end of buffer.
;  20030121:	dkres: fixed bug allowing tab completions to extend command
;		line beyond end of buffer.
;  20030121:	dktab: remove trailing backslash if tab completion session
;		ends w/a CR  (for easier use with CD command).
;  20031022:	allowed chars 7f-ff to be part of filename/path.  This fixes
;		problems with umlauts.
; --------------
; Stuff to do:
;
; *)	Implement Win 95 extensions.
; *)	Search MACROS for command completion first.
; *)	Search internal commands during command completion.
; *)	Simple compression of string lists? (macros, history, help msg, etc)?
; *)	Auto-convert commands "macroname=xxx" to "doskey macroname=xxx".
; *)	DOSKEY @macro=file	(define multi-line macro)
;	DOSKEY @=file		(define immediate exec macro, perhaps use
;				 this to implement startup multi-macro def).
; *)	Option not to install stupid "pick command line by number" option.
; *)	Move new macro def to macro buf manually, then make all macro ops
;	near (es==ds).
; *)	Whole screen display subsystem is a mess - redesign.
; *)	Implement function key=macro logic (allow func key redefinition).
;
; Functional differences from Microsoft DOSKEY.
;
;  (1)	No DOS version check.
;  (2)	Closed standard handles (STDOUT, STDIO, etc) before TSR installation.
;	This was done to prepare for (3), since to unload I simply release
;	the TSR memory block - this would leave the standard handles open.
;  (3)	Implemented '/UNINSTALL' feature/switch that unloads DOSKEY from
;	memory.  Not all that useful in practice, but doesn't cost anything
;	in installed space, and sure is handy during development (so I don't
;	need to reboot to test the next program change).
;  (4)	Command line options can begin with '-' as well as '/'.
;  (5)	No bells (in error messages, or during command line overflow).
;  (6)	ALTF8 works as F8, only forward instead of reverse.
;  (7)	F10 displays macros.
;  (8)	ESCAPE resets command history pointer to bottom of list.
;  (9)	Tab/Shift+tab completion.
; (10)	ALTF6 generates a tab character.
;
;-----------------------------------------------------------------------------

	.nolist
	include	dkdefs.inc	;include constant definitions, externdef's
	.list
	option	noljmp		;disallow automatic jmp-lengthening

;-----------------------------------------------------------------------------
;
; Define entry point, which immediately jumps to the exec handling code at
; the end of the image.
;
; Note the "near ptr" is needed to get the correct "1st pass" size of this
; jump (so the sjmpend offset is correct).

	org	100h		;org to define entry jmp
start	label	near
	jmp	near ptr Startup ;jump to exec code at end of image
sjmpend	equ	$

;-----------------------------------------------------------------------------
;
;	<<<<<<<<  Start of installed variable declarations  >>>>>>>>
;
; *** NOTE: changing/adding/moving/redefining variables in ANY WAY	***
; *** necessitates the modification of DOSKEY_ID; otherwise, an attempt	***
; *** to use the new DOSKEY command to update/interrogate an installed	***
; *** TSR would be fatal.						***
;
; We utilize the PSP to hold run-time buffers and variables, and include it
; as part of the instance data (this keeps us from otherwise wasting this
; portion of the installed image).  Fields declared in the PSP cannot be
; initialized at compile-time, and cannot be used at startup time (by the
; exec code), since the PSP fields they overlay (incoming command line,
; environment address, etc.) need to be accessed during startup.
;

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Start of uninitialized fields that overlay the first 128 bytes of
	; the PSP.  These fields should be used by the resident code only.

	org	0
SwiData0Start label	near	;start of first instance data block

R2SIZE	equ	sizeof PSP.pspReserved2 ;(masm complains unless this is equ'd)
	if	R2SIZE gt sizeof ENTRYSTACKFRAME
	  .err	<EntryFrame won't fit in PSP.pspReserved2>
	endif
	org	PSP.pspReserved2
EntryFrame ENTRYSTACKFRAME <>	;entry int 2fh stack frame moved here

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Overlay run-time data areas with (unused) PSP FCBs.
	;
	; The int 2fh address should be kept fixed at offset PSP.pspFCB_1.
	; This is critical for development only - we can unload a modified
	; version (with a new code and/or data layout) with it fixed;
	; otherwise, the int 2fh vector restore could be fatal.

	org	PSP.pspFCB_1
Old2fVector dd	?		;pre-installation int 2f vector
slMacros  STRINGLIST <>		;macro string list control structure
slHistory STRINGLIST <>		;command history list control structure
scInfo	  SCREENINFO <>		;screen information

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Command buffer control fields.  When editing is underway, cbCurrent
	; is the address of the cursor location, and cbLast is the address
	; beyond the last Template byte that has been made officially part
	; of CmdBuf via an editing key (like F1).
	; cbActive is not used during editing.  After editing is complete,
	; it is pointed to the start of the command, which is executed
	; from there.  If on re-entry cbActive points to a non-zero (the
	; prior command didn't fully execute, or was a macro), cbActive is
	; assumed to point within CmdBuf, and CmdBuf execution is continued
	; from that point (note: if cbActive does point to a zero, that zero
	; byte may not be within CmdBuf).

cbCurrent dw	?		;current (cursor) address within CmdBuf
cbLast	  dw	?		;CmdBuf highwater offset of Template
cbActive  dw	?		;points to non-zero when CmdBuf macro "active"

FCBend	equ	$

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; CmdBuf fields are placed at offset PSP.pspCommandTail.  This
	; is as if it were an incoming command line (for no real reason,
	; except that it makes things a bit easier to debug).
	;
	; Note the extra CmdBuf byte (129 total instead of 128) - it could be
	; accessed during temporary termination, so leave it there.
	;
	; The CmdBufLen is not used during command input.

	org	PSP.pspCommandTail
	if	$ lt FCBend
	  .err	<pspFCB overlay area too large>
	endif
CmdBufLen	db	?	;CmdBuf len prefix *** MUST PRECEDE CmdBuf ***
CmdBuf		db 127 dup (?)	;command input buffer
CmdBufLimit	db   2 dup (?)	;CmdBuf limit (last possible terminator slot)


tflags	  db	?		;run-time flags

	if	$ ne sjmpend
	  .err	<sjmpend/tflags mismatch>
	endif

;-----------------------------------------------------------------------------

NumberBuffer	db	8 dup(0)

SwiData0End	label	near	;#############################################

Constant10	dw	10		;for use by mul/div opcodes

MoreMsg		db	CR,'-- More --',0

	;non-filename/path characters >20h and <7fh (i.e., in printable range)
SeparatorList	db	'"*,/;<=>?[]^|'
CommandPrefix	db	' ,=;'		;characters allowed to prefix command
CommandPrefixLast db	TAB		;last CommandPrefix byte
PathEquals	db	'path=',0	;used to scan environment for path
CommandExts	db	'.com',0	;recognized command extension table
		db	'.bat',0
		db	'.exe',0
		db	0		;(terminate CommandExts table)

DollarSrc db	'tlgb$',CMDSEP,0	;must be in descending binary value
DollarSub db	CMDSEP,'<>|$',CMDSEP,0	;characters to substitute for above

StarDotStar	db	'*.*'		;used to terminate find first masks
SingleZero	db	0		;single zero/StarDotStar terminator

LinePrompt	db	'Line number: '	;used for history row selection (F9)

;-----------------------------------------------------------------------------
; The below mess defines the keystroke dispatching tables.  All the macro
; nonsense is simply to allow the pairing of the keystroke and handler
; address via DKEY invocations.

DKEY	macro	a,a1,b,b1,c,c1,d,d1,e,e1,f,f1,g,g1,h,h1
	ifnb	<a>
	  DKEY2	%(a),a1
	  DKEY	b,b1,c,c1,d,d1,e,e1,f,f1,g,g1,h,h1
	endif
	endm
DKEY2	macro	keystroke,dispatch
	ifb	KeyString
KeyString textequ <db >,<keystroke>
	else
KeyString catstr KeyString,<,keystroke>
	endif
	dw	dispatch
	endm

tablejmp label	word
KeyString textequ <>
 DKEY	CR,FinishLineAndExit,	LF,DoNothing  ;021018, CTL_F,DoNothing ;021018
 DKEY	ESCAPE,ClearCmdBuf,	BS,BackSpace
 DKEY	TAB,TabComplete
 DKEY	5,GotoEndOfLine,	1,GotoStartOfLine	;basic "emacs" control
 DKEY	2,BackupChar,		6,ForwardChar		;sequences added for
 DKEY	11,DeleteAfter,		4,DeleteChar		;  Scott Baker 021018
t1str	textequ KeyString
KeyString textequ <>
 DKEY	0,StoreChar
 DKEY	F1,ForwardChar,		RIGHT,ForwardChar,	INSKEY,ToggleInsert
 DKEY	LEFT,BackupChar,	F6,GenerateCtlZ,	HOME,GotoStartOfLine
 DKEY	DEL,DeleteChar,		CTLEND,DeleteAfter,	CTLHOME,DeleteBefore
 DKEY	CTLLEFT,BackupWord,	CTLRIGHT,ForwardWord,	F4,DeleteToChar,
 DKEY	F2,CopyTemplateToChar,	F5,CopyAllToTemplate,	F3,CopyToTemplateEnd
 DKEY	ENDKEY,GotoEndOfLine,	DOWN,SelectNext,	UP,SelectPrevious
 DKEY	PAGEUP,PageUp,		PAGEDOWN,PageDown,	F7,DisplayHistory
 DKEY	F10,DisplayMacros,	ALTF10,ClearMacros,	ALTF7,ClearHistory
 DKEY	F8,ExtendLineUp,	ALTF8,ExtendLineDown,	F9,EnterCommandNumber
 DKEY	ALTF6,GenerateTab
 dw	DoNothing		;*** default dual-byte handler
table1	t1str
table2	KeyString
table2end label byte		;end of dual-byte table

;-----------------------------------------------------------------------------

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Task switcher startup info structure, and instance data block
	; descriptor list.  The instance data block list contains contiguous
	; descriptors that describe blocks of memory that a task switcher
	; (e.g., Windows) will keep unique between multiple DOS sessions.
	;
	; We have two such blocks of memory:
	;
	;   1) Generic variables, from SwiData0Start to SwiData0End.
	;   2) Block of memory that will hold macros and history.
	;
	; The portions of the instance data block descriptor structures that
	; cannot be filled in at assembly time are initialized during init.

SwiData0 SWINSTANCEITEM <offset SwiData0Start, SwiData0End - SwiData0Start>
SwiData1 SWINSTANCEITEM <offset EndResident, DEFAULT_BUFSIZE>
	 dd	0		;terminate SWINSTANCEITEM list

				;switcher startup info structure
SwStartup SWSTARTUPINFO <3, 0, 0, 0, offset SwiData0>

;-----------------------------------------------------------------------------
;
; Get command line, either input by user, or expanded from a continuing
; command line or macro execution.
;
; IN:
;  interrupts are enabled
;  EntryFrame.TemplatePtr= address of incoming command line buffer.  First
;	byte is the incoming buffer size, which has been verified externally
;	to be exactly IOBUF_SIZE (128).  This byte is not examined or changed
;	here.
;
;	The 2nd byte holds the size of the incoming command buffer, which is
;	used as the editing template (e.g., for the F1 key).  Up to a max of
;	CMDLINE_MAX (127) characters are copied from the incoming buffer to
;	our CmdBuf, for use as the template.
;
;	On exit, the 2nd byte will receive the returned input size, not
;	counting the CR attached to the end of the input.  This will be zero
;	if a macro is entered - the actual macro command(s) will be returned
;	in subsequent call(s).
;
;	The third byte and beyond hold the actual input: there will be a max
;	of IOBUF_SIZE - 1 input bytes returned, followed by a single CR
;	character.
; OUT:
;	Template[0]= unchanged (never examined - see above).
;	Template[1]= cmd length, not including terminating CR.
;		If a macro is entered, this will be zero.  The next input
;		call will return the first command line for the macro.
;	Template[2]= first command line character - CR if returned line empty.

GetCmdLine	proc	near

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; First check if we're executing a macro.  If so, continue its
	; execution.

	mov	si,slMacros.Active	;si= current active macro address
	.if	byte ptr [si] != 0	;if a macro is executing,
	  mov	cx,IOBUF_SIZE - 1	;cx= max chars that can be returned
	  les	di,EntryFrame.TemplatePtr ;point beyond returned buffer sizes
	  inc	di
	  inc	di

	  .while 1			;loop to process macro characters,
	    lodsb			;get next macro char
	    .if	al == '$'		;if $x translation needed,

	      lodsb			;get next byte
	      .if al <= '9'		;if may be a parameter transfer,
		.if al == '*' || al >= '1' ;if valid parameter index,
		  call TransferMacroParameter ;transfer the parameter
		  .continue		;continue macro execution
		.endif
	      .endif

	      call ToLower		;convert to lowercase
	      mov bx,offset DollarSrc - 1 ;descending list of $ pairings, 0
	      .repeat			;scan for mapped pairing
		inc bx			;next valid pairing
	      .until al >= [bx]
	      .continue .if !zero?	;continue= invalid pairing, ignore
	      mov al,(DollarSub - DollarSrc)[bx] ;map to substitution char
	    .endif

	    .break .if al == 0		;break= end of macro, exit
	    .break .if al == CMDSEP	;break= command separator, exit

	    jcxz @F			;ignore if output buffer is full
	      stosb			;transfer macro char to command
	      call DisplayFormattedAl	;(display chars as we transfer them)
	      dec cx			;reduce size of output buffer
	    @@:
	  .endw

	  .if	al == 0			;if hit macro terminator above
	    dec  si			;don't go beyond macro terminator
	    mov  bx,cbActive		;discard command used for argument(s)
	    .repeat
	      mov al,[bx]
	      .break .if al == 0	;don't go beyond end of command
	      inc bx
	    .until al == CMDSEP
	    mov  cbActive,bx
	  .endif
	  mov	slMacros.Active,si	;update active macro pointer
	  jmp short TerminateCommand
	.endif

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Macro not executing - either continue execution of a Ctl<T>
	; separated command line (if one is pending), or prompt user for a
	; new entry.

	mov	si,cbActive		;CmdBuf execution pointer
	mov	bx,offset DisplayFormattedAl ;assume echo as we transfer
	.if	byte ptr [si] == 0	;if CmdBuf inactive (not executing),
	  call	EditCmdBuf		;get new user entry
	  mov	si,offset CmdBuf	;see if new entry a multi-part command
	  .repeat
	    lodsb
	    cmp al,CMDSEP		;multi-part command?
	    jz  TerminateEmpty		;jmp= yes, execute it like a macro
	  .until al == 0
	  mov si,offset CmdBuf		;reset ptr to beginning of command
	  mov bx,offset DoNothing	;no re-display of just-entered command
	.endif

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; If a macro is found at the start of the command, 'activate' the
	; macro (put us in "macro execution" state).  This causes an empty
	; command line to be returned for this call.  On the next call, the
	; first command spawned by the macro will be returned.

	call	ActivateMacro		;try to parse-off/activate macro
	jz	TerminateEmpty		;jmp= execute macro next time in

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Execute next command from CmdBuf.

	mov	cx,IOBUF_SIZE - 1	;cx= max chars that can be returned
	les	di,EntryFrame.TemplatePtr ;point to return buffer
	inc	di
	inc	di
	.while	1
	  mov	al,[si]			;stop if we've hit terminator
	  .break .if al == 0
	  inc	si			;bypass character
	  .break .if al == CMDSEP	;stop if command separator hit
	  jcxz @F			;ignore if output buffer is full
	    stosb			;transfer macro char to command
	    call bx			;(display chars, unless inhibited)
	    dec cx			;reduce size of output buffer
	  @@:
	.endw
	mov	cbActive,si		;update CmdBuf execution pointer
	jmp short TerminateCommand	;jmp= terminate command and exit

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Here to terminate with an empty command line.

TerminateEmpty:

	les	di,EntryFrame.TemplatePtr ;exit, returning null command
	inc	di
	inc	di

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Here to terminate a command and exit.

TerminateCommand:

	mov	al,CR			;terminate generated command
	stosb
	mov	ax,word ptr EntryFrame.TemplatePtr
	xchg	ax,di			;ax= length of command (without CR)
	inc	di
	sub	ax,di
	sub	al,2
	stosb				;return length of generated command

	ret
GetCmdLine	endp

;-----------------------------------------------------------------------------
; Here to substitue a $1-$9 or $* macro parameter.
;
; IN:
;   al= parameter indicator (*,0-9)
;   cx= remaining output buffer space
;   es:di= output buffer pointer

TransferMacroParameter proc near

	sub	al,'*'			;assume '*'
	.if	!zero?			;if assumption wrong,
	  sub	al,'0' - '*'		;convert to 1-9
	.endif
	cbw
	xchg	bx,ax

	push	si			;save current macro address
	mov	si,cbActive		;si= command buffer parameters

	.while	1			;loop to find selected parameter,
	  call	SkipWhitespace		;skip whitespace in front of parameter
	  dec	bx			;break= selected parameter found
	  jle @F;.break .if sign? || zero?
	  .repeat			;skip to start of next parameter
	    mov	al,[si]			;get suspected parameter byte
	    .break .if al == 0 || al == CMDSEP ;don't skip beyond command end
	    inc	si			;advance beyond parameter byte
	    call IsWhitespace
	  .until zero?			;loop until end of parameter
	.endw
	@@:

	.while	1			;loop to transfer parameter
	  lodsb				;get next parameter byte
	  .break .if al == 0 || al == CMDSEP ;break= end of command line, exit
	  or	bx,bx			;transferring entire line?
	  .if	!sign?			;if not,
	    call IsWhitespace
	    .break .if zero?		;break= end of parameter, exit
	  .endif
	  jcxz	@F			;ignore if output buffer is full
	    stosb			;transfer macro char to command
	    call DisplayFormattedAl	;(display chars as we transfer them)
	    dec	cx			;reduce size of output buffer
	  @@:
	.endw

	pop	si			;restore current macro address

	ret
TransferMacroParameter endp

;-----------------------------------------------------------------------------
;
; Here we edit the command line on the screen, and add it to the history.

EditCmdBuf	proc	near

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; The first thing we do is initialize the SCREENINFO structure
	; (screen/cursor size, entry/current cursor location).
	;
	; This information is needed for repositioning the cursor during
	; deletion, -- more -- messages, toggling insert mode, etc.
	;
	; First try to get the number of screen rows from ansii.sys.  This
	; requires a DISPLAYMODE buffer - get scratch space for it from
	; the history string list pool.
	;
	; Note: we use the STDERR device in the expectation that there's
	; less chance that it's been redirected.

	mov	cx,sizeof DISPLAYMODE	;di= get freespace for DISPLAYMODE
	call	GetHistoryFreespace
	assume	di:near ptr DISPLAYMODE
	mov	ax,14			;(size of fields after dmDataLength)
	mov	[di].dmInfoLevel,ah	;init DISPLAYMODE for get info call
	mov	[di].dmDataLength,ax

	mov	dx,di			;get ansi.sys char device info (all
	mov	cx,37fh			;  we're interested in is the number
	mov	bx,STDERR		;  of screen rows)
	mov	ax,440ch
	int	21h
	mov	ax,[di].dmRows		;assume success, load number of rows

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; If ansii.sys failed, or we're in graphics mode, try to get number
	; of screen rows from the system BIOS.  If this looks wrong, just use
	; 25.

	.if	carry? || [di].dmMode != 1 ;if error, or not in text mode,
	  mov	al,25			;assume 25 (can't get from BIOS)
	  xor	cx,cx			;fetch BIOS video rows
	  push	ds
	  mov	ds,cx
	  mov	cl,ds:VIDEO_ROWS	;get video row count - 1
	  pop	ds
	  jcxz	@F			;jmp= value unreliable
	    xchg ax,cx			;value may be good, use it
	    inc	 ax
	  @@:
	.endif
	assume	di:nothing
	mov	scInfo.TotalRows,al

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Get number of screen columns and incoming cursor location and mode
	; from the system BIOS.

	mov	ah,0fh			;ah= number of screen columns
	int	10h			;bh= current video page
	mov	scInfo.TotalCols,ah
	mov	ah,3			;get current cursor location
	int	10h
	xchg	ax,dx
	mov	word ptr scInfo.EntryCol,ax ;ah= entry row, al= entry column
	mov	word ptr scInfo.CurrentCol,ax
	mov	scInfo.EntryMode,cx	;ch= cursor start, cl= cursor end

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Initialize the flags.

	mov	al,tflags		;al= flags
	and	al,not (TF_INSERT or TF_EXIT)	;kill insert, exit request
	.if	al & TF_STARTINSERT	;if insert needs to be on
	  or	al,TF_INSERT		;turn it on
	.endif
	mov	tflags,al		;save initial flags

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Set the current insert mode, and corresponding cursor size.

	call	InitInsertMode		;set cursor mode appropriately

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Init CmdBuf to template contents, and edit it.

	call	CopyTemplateToCmdBuf
	.repeat
	  call	GetKeybCharNoEcho
	  call	DispatchKeystroke

	  call	QueryClose		;close DOS box request sensed?
	  .if	zero?			;if yes,
	    call DeleteAll		;erase entire line,
	    .break			;  and exit keystroke entry loop
	  .endif

	.until	tflags & TF_EXIT	;loop until exit requested

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Restore entry cursor mode.

	mov	cx,scInfo.EntryMode	;restore entry cursor mode
	mov	ah,1
	int	10h

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Discard unused template characters (terminate transferred chars
	; with a zero), and add command to history if it is non-empty.

	mov	si,offset CmdBuf	;si= start of user input
	mov	cbActive,si		;point to command to execute
	mov	di,cbLast		;di= end of user input
	mov	byte ptr [di],0		;discard unused template characters
	.if	si != di		;if entry non-empty,
	  assume bx:near ptr STRINGLIST
	  mov	bx,offset slHistory	;store entry to command history
	  call	StringListAppend
	  mov	ax,[bx].Tail		;set selected past bottom of history
	  mov	[bx].Active,ax
	  assume bx:nothing
	.endif

	ret
EditCmdBuf	endp

;-----------------------------------------------------------------------------

ToggleInsert	label	near
	xor	tflags,TF_INSERT	;invert current insert state

InitInsertMode proc	near

	mov	cx,scInfo.EntryMode	;cx= cursor mode on entry
	mov	al,tflags		;al= current flags

	push	ds			;prepare to access BIOS shift status
	xor	bx,bx
	mov	ds,bx
	mov	bx,BIOS_SHIFT_STATUS
	and	byte ptr [bx],not SS_INSERT ;assume insert off
	.if	al & TF_INSERT		;if assumption wrong,
	  sub	ch,2			;set insert on (widen cursor)
	  .if	carry?
	    mov	ch,0
	  .endif
	  or	byte ptr [bx],SS_INSERT
	.endif
	pop	ds

	mov	ah,1		; Video display   ah=functn 01h
	int	10h		;  set cursor mode in cx

	ret
InitInsertMode endp

;-----------------------------------------------------------------------------
;
; ax = keystroke.
; ah= 0 for ascii al, ah= 1 for dual-byte (extended) code in al.
;
; No other registers altered.
; 
; Returns non-zero if character is a dual-byte keystroke (ah= 1).

KeybProc proc	near private

	.repeat
			; No keystroke available, allow other system operation
	  int	28h			;allow TSR operation (keyboard idle)
	  mov	ax,1680h		;MS-DOS idle call (same as int 28h)
	  int	2fh

GetKeybCharNoEcho label	near
	  call	QueryClose		;close DOS box request sensed?
	  mov	ax,ESCAPE		;assume yes, set up ESCAPE keystroke
	  jz	@F			;jmp= exit if close DOS box sensed

	  mov	ah,0bh			; DOS Services  ah=function 0Bh
	  int	21h			;  if keybd char available,al=FF

	.until	al != 0
	mov	ah,8			; DOS Services  ah=function 08h
	int	21h			;  get keybd char al, no echo
	@@:

	and	ax,0ffh			;assume single byte character
	.if	zero?			;if assumption wrong
	  mov	ah,8			;get 2nd byte
	  int	21h
	  cmp	al,0			;shouldn't happen, but would be fatal
	  jz	GetKeybCharNoEcho	;jmp= guarantee NEVER return al == 0
	  mov	ah,1			;indicate dual-byte code
	.endif

	or	ah,ah			;non-zero if dual-byte keystroke
	ret
KeybProc endp

;-----------------------------------------------------------------------------
; Test for external DOS box close (Windows '95 extension).
;
; IN:
;   none
; OUT:
;   zero status if DOS box close requested
;   ax destroyed; all other registers preserved

QueryClose proc	near
	push	dx
	mov	dx,0100h		;Win '95 extension: query close
	mov	ax,168Fh
	int	2fh			;check for external close request
	pop	dx
	test	ax,ax			;zero status if close request sensed
	ret
QueryClose endp

;-----------------------------------------------------------------------------
; IN:
;   si= start of potential macro name
;
; OUT:
;   non-zero status if macro not matched:
;	cbActive= unchanged
;	slMacros.Active= unchanged
;
;   zero status if macro was matched and activated:
;	cbActive= set to beyond macro name in source (si)
;	slMacros.Active= set to definition of macro (beyond =)
;
;   si, bx= preserved

ActivateMacro	proc	near
	push	si

	.repeat				;dummy block (to support breaks)
	  mov	di,si			;save start of potential macro name
	  lodsb				;zero if 1st char valid as macro name
	  call	IsMacroNameChar
	  .break .if !zero?		;break= no macro found

	  .repeat			;scan to end of macro name
	    lodsb
	    call IsMacroNameChar
	  .until !zero?
	  dec	si			;point to name terminator

	  push	word ptr [si]		;terminate name (for searching)
	  push	si
	  mov	word ptr [si],'='
	  mov	si,di			;search for macro definition
	  mov	di,slMacros.Head
	  call	StringListPoolSearch
	  pop	si			;undo early termination
	  pop	word ptr [si]
	  .break .if !zero?		;break= no match
	  mov	cbActive,si		;save address beyond matched name
	  mov	slMacros.Active,di	;save address of macro definition text
	.until	1			;end dummy block (to support breaks)

	pop	si
	ret
ActivateMacro	endp

;-----------------------------------------------------------------------------
; IsMacroNameChar:
;	Sets zero status if al is a valid macro name character.
;
; IsPathChar:
;	Sets zero status if al is a valid file/path character.
;
; IsWhitespace:
;	Sets zero status if al is whitespace.
;
; al= unchanged

	.repeat
	  test	al,al			;not macro name char - ret w/nz status
	  ret
IsMacroNameChar	label	near

	  .if	al != ':'		;zero status if not a macro name char
	    cmp	al,'\'
	  .endif
	.until	!zero?

IsPathChar proc near		;zero status if al is a file/path character
	push	di
	push	cx
	mov	cx,sizeof SeparatorList + 1 ;cx= separator count + 1
;031022	.if al > 20h && al < 7fh	;if printable, may be file/path char,
	.if al > 20h			;only allow chars >20h,		;031022
	  mov	di,offset SeparatorList	;list of separator chars
	  repne scasb			;cx= 0 if al not found in list
	.endif
	test	cx,cx			;zero status if file/path character
	pop	cx
	pop	di
	ret
IsPathChar endp

IsWhitespace	proc	near
	.if	al != ' '
	  cmp	al,TAB
	.endif
	ret
IsWhitespace	endp

;-----------------------------------------------------------------------------
;
; Skips whitespace at ds:si.  si is left pointing at 1st non-whitespace.
;
;  al is set to character si is left pointing to.
;  all other registers saved.

SkipWhitespace proc	near

	.repeat
	  lodsb
	.until	al != ' ' && al != TAB
	dec	si

	ret
SkipWhitespace	endp

;-----------------------------------------------------------------------------
;
; Convert al to lowercase

ToLower	proc	near
	.if	al >= 'A' && al <= 'Z'
	  add	al,'a' - 'A'
	.endif
	ret
ToLower	endp

;-----------------------------------------------------------------------------
; Clear all History.
;
; Clear all macros.
;
; Clear history is trivial.  Clear macros requires we shrink the macro
; string list pool to 1 byte (and appropriately enlarge the history string
; pool).

ClearMacros proc	near
	assume	bx:near ptr STRINGLIST
	mov	bx,offset slMacros
	mov	ax,[bx].Next		;ax= last macro area byte
	dec	ax
	mov	[bx].Head,ax		;shrink macro area to 1 byte
	mov	slHistory.Tail,ax	;enlarge history region appropriately
	.if	0
ClearHistory label near
	  mov	bx,offset slHistory
	.endif
	jmp	StringListClear
	assume	bx:nothing
ClearMacros endp

;-----------------------------------------------------------------------------
; si, bx, dx= preserved

BackupActiveHistory	proc	near
	mov	di,slHistory.Active	;di= selected history line
	mov	cx,di			;cx= history bytes that precede it
	sub	cx,slHistory.Head
	std				;scan to start of previous entry
	dec	di
	dec	di
	mov	al,0
	repne scasb			;(note: may read 1 byte BEFORE start -
	inc	di			;  that's OK, it's in our segment)
	inc	di
	cld
	mov	slHistory.Active,di	;save new selected entry
	ret
BackupActiveHistory	endp

;-----------------------------------------------------------------------------
; cx= length of active history row w/o zero terminator (0 if at end of list)
; di= points beyond terminator
; si, bx, dx= preserved

GetActiveHistoryRowLength proc near
	mov	di,slHistory.Active	;di= selected history line
	mov	cx,-1			;scan past end of entry
	mov	al,0
	repne scasb
	inc	cx			;cx= count of chars, w/o terminator
	not	cx
	ret
GetActiveHistoryRowLength endp

;-----------------------------------------------------------------------------
; si, bx, dx= preserved

AdvanceActiveHistory	proc	near
	call	GetActiveHistoryRowLength ;cx= active history row length
	jcxz	@F			;jmp= no active row (at end of list)
	  mov	slHistory.Active,di	;advance active beyond current row
	@@:
	ret
AdvanceActiveHistory	endp

;-----------------------------------------------------------------------------

ClearCmdBuf label near
	mov	ax,slHistory.Tail	;reset history to bottom of list
	mov	slHistory.Active,ax	;(DOSKEY doesn't do this, but should)

	call	DeleteAll		;erase entire line

; NOTE: fall through to CopyTemplateToCmdBuf...

;-----------------------------------------------------------------------------
;
; Resets command buffer to contents of incoming template.  Also initializes
; the CmdBuf control variables (cbxxx).

CopyTemplateToCmdBuf	proc	near
	mov	si,offset CmdBuf
	mov	di,si
	call	Set_cbCurrent_cbLast	;set new cb pointers
	push	ds
	lds	si,EntryFrame.TemplatePtr
	xor	ax,ax
	inc	si
	lodsb
	.if	ax > CMDLINE_MAX
	  mov	ax,CMDLINE_MAX
	.endif
	xchg	cx,ax

	jcxz	@F
	  .repeat
	    lodsb
	    .break .if al == 0	;(0's would be fatal)
	    .break .if al == CR
	    stosb
	  .untilcxz
	@@:
	pop	ds
	xor	ax,ax		;indicate not in macro execution state
	stosb			;terminate CmdBuf
CopyTemplateToCmdBuf	endp

; NOTE: fall through to cbActiveDisable...

;-----------------------------------------------------------------------------
; Disables any "active" command by pointing cbActive to a zero.  Note the zero
; we point to is NOT within CmdBuf - the fact that *cbActive is zero is all
; that is required.

cbActiveDisable	proc	near
	mov	cbActive,offset SingleZero
	ret
cbActiveDisable	endp

;-----------------------------------------------------------------------------

DispatchKeystroke proc	near

	mov	di,offset table1	;assume single byte character
	mov	cx,(table2 - table1) + 1
	.if	ah != 0			;if assumption wrong,
	  mov	di,offset table2 + 1	;use dual-byte character table
	  mov	cl,(table2end - (table2 + 1)) + 1
	.endif
	repne scasb			;scan selected table
	sub	di,offset table1 + 1	;di= jump table index
	add	di,di
	jmp	tablejmp[di]

DispatchKeystroke endp

;-----------------------------------------------------------------------------
; gets	si= cbCurrent
;	di= cbLast
;
; no other registers altered

Get_cbCurrent_cbLast	proc	near	;load cb Current/Last pointers
	mov	si,cbCurrent
	mov	di,cbLast
	ret
Get_cbCurrent_cbLast	endp

;-----------------------------------------------------------------------------
; sets	cbCurrent= si
;	cbLast= di
;
; If si > di, di set to si
;
; no other registers altered

Set_cbCurrent_cbLast proc near	;store cb Current/Last pointers
	.if	si > di		;advance template inclusion ptr, if needed
	  mov	di,si
	.endif
	mov	cbCurrent,si
	mov	cbLast,di
	ret
Set_cbCurrent_cbLast	endp

;-----------------------------------------------------------------------------

FinishLineAndExit proc	near		;handles entry of a CR
	call	Get_cbCurrent_cbLast	;get cb pointers
	mov	byte ptr [di],0		;terminate entry (truncate template)
	call	DisplayFormattedMsg	;display to move to end of entry
;extra linefeed generated, is the CR needed???
;???	mov	al,CR			;move to left margin
;???	call	DisplayFormattedAl

ExitKeystrokeDispatcher label near
	or	tflags,TF_EXIT		;force keystroke dispatcher exit

DoNothing label	near
	ret
FinishLineAndExit endp

;-----------------------------------------------------------------------------

GenerateTab label near
	mov	al,TAB
	skip
GenerateCtlZ label near
	  mov	al,CTL_Z
	endskip

; NOTE: fall through to StoreChar...

;-----------------------------------------------------------------------------
; StoreChar - stores char in al according to current insert mode.
; StoreCharInsertMode - inserts char in al, regardless of current insert mode.
; StoreStringInsertMode - inserts asciiz string at si (regardless of
;	insert mode).  Char(s) that won't fit are ignored.
;
; bx preserved, all other regs destroyed

StoreChar proc	near
	mov	cl,tflags		;get flags (for insert state)
	skip
StoreCharInsertMode label near
	  mov	cl,TF_INSERT		;force insert mode
	endskip
	mov	si,offset SingleZero	;only store/insert the char in al
	.if	0
StoreStringInsertMode label near
	  lodsb				;al= 1st byte to insert
	  mov	cl,TF_INSERT		;force insert mode
	.endif

	push	bx

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; First update CmdBuf by inserting/overwriting the new data.  The
	; screen will be updated later.

	mov	bx,si			;fetch additional source chars via bx
	call	Get_cbCurrent_cbLast	;get cb pointers
	mov	dx,offset CmdBufLimit	;limit for terminator
	and	cx,TF_INSERT		;cx= non-zero if inserting
;030121	push	si			;save entry cursor location
	.while	al != 0			;while more char(s) to store,
	  jcxz	@F			;if inserting,
	    .break .if di == dx		;break= buffer full, stop inserting
	    push si
	    .repeat			;insert new char into buffer
	      xchg [si],al
	      inc  si
	    .until al == 0 || si == dx	;until terminator, or buffer limit
	    mov	[si],ch			;re-terminate
	    pop	si
	    inc di			;advance high water mark	;030121
	  .if	0			;else overwriting,
	    @@:
	    .break .if si == dx		;break= buffer full, stop overwriting
	    mov	ah,[si]			;load char being overwritten
	    mov	[si],al			;overwrite char
	    .if	ah == 0			;if extending buffer,
	      mov [si+1],ah		;re-terminate
	    .endif
	  .endif
	  inc	si			;advance storage ptr
	  mov	al,[bx]			;fetch next char to store
	  inc	bx
	.endw
	mov	bx,si			;bx= end of insertion
;030121	pop	si			;restore entry cursor location
	call	Get_cbCurrent_cbLast	;reload original cb pointers	;030121

	sub	bx,si			;bx= count of stored chars (0-n)
	jz	StoreCharRet

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; If there is something onscreen beyond the current cursor position
	; (i.e., we're not simply extending the command line), we have to
	; adjust the display accordingly - we can't simply display the new
	; chars, even if overwriting, since the display widths may differ.

	push	bx			;save count of stored char(s)
	.if	si != di		;if we updated screen beyond cursor,
	  mov	al,ah			;assume we didn't insert
	  dec	bx
	  jcxz	@F			;jmp= assumption correct
	    inc	bx			;undo assumption, restore insert size
	    add	di,bx			;insertion always advances cbLast
	    mov	cbLast,di
	    mov	al,[si+bx]		;set up for OverwriteFormattedMsgAl
	  @@:
	  push	word ptr [di]		;save char beyond end of display
	  push	di
	  mov	[di],ch			;terminate displayed text
	  lea	di,[si+bx]		;update display (cursor and beyond)
	  call	OverwriteFormattedMsgAl
	  pop	di			;restore char beyond end of display
	  pop	word ptr [di]
	.endif
	pop	cx			;cx= count of stored char(s)
	call	ForwardCxChars		;jmp= advance cursor

StoreCharRet:
	pop	bx
	ret
StoreChar endp

;-----------------------------------------------------------------------------
;
; Advances current cursor location 1 (or cx) CmdBuf char(s).  Advancement
; stops if end-of-buffer is hit.
;
; OUT:
;   si= current cursor address (advanced by chars we moved forward)
;   di= last onscreen address (potentially modified by advancement)
;   cx(ForwardCxChars only)= count of chars to move forward
;   ax/cx= destroyed
;   dx= count of chars forwarded (0-n)
;   all other registers preserved
;
;   zero status if no change to cursor location occurred

ForwardChar proc near
	mov	cx,1			;forward 1 char
ForwardCxChars label near
	call	Get_cbCurrent_cbLast	;get cb pointers
	xor	dx,dx			;init count of amount forwarded
	jcxz	@F			;jmp= nothing to advance
	  .repeat			;for all chars to advance,

;011212	    lodsb			;get char to advance over/display
;011212	    .if al			;if not at end of entry/template
	    .if byte ptr [si] != 0	;if not at end of entry/template,
	      lodsb			;get char to advance over/display

	      call DisplayFormattedAl	;display char, advance cursor
	      inc  dx			;indicate advance was performed
	    .endif
	  .untilcxz
	@@:
	call	Set_cbCurrent_cbLast	;set new cb pointers
	or	dx,dx			;set return flags
	ret
ForwardChar endp

;-----------------------------------------------------------------------------
;
; Backs up current cursor location 1 or more chars.
;
; OUT:
;   si= current cursor address (either unchanged, or backed-up by 1)
;   zero status if no change to cursor location occurred.

BackupChar proc near
	mov	cx,1			;back up 1 char
BackupCxChars label near
	call	Get_cbCurrent_cbLast	;get cb pointers

	mov	ax,offset CmdBuf	;(can't back up before here)
	mov	di,si			;di= potential new cbCurrent, after
	sub	di,cx			;  attempting to back up cx chars
	.if	carry? || di < ax	;if backed up in front of left margin,
	  xchg	di,ax			;limit to left margin
	.endif
	.if	di != si		;if backing up at least 1 char,
	  push	word ptr [di]		;save char under cursor, and location
	  push	di
	  mov	byte ptr [di],0		;terminate at new cursor location
	  mov	ax,word ptr scInfo.EntryCol ;set cursor to entry location
	  mov	word ptr scInfo.CurrentCol,ax
	  mov	si,offset CmdBuf	;display preceding chars virtually
	  call	DisplayFormattedMsgVirtual
	  call	SyncCursor		;sync onscreen/scInfo cursor location
	  pop	si			;restore character under cursor
	  pop	word ptr [si]
	  mov	cbCurrent,si		;save new current CmdBuf pointer
	  or	al,0ffh			;indicate cursor location was moved
	.endif

	ret
BackupChar endp

;-----------------------------------------------------------------------------

GotoStartOfLine	proc	near
	mov	cbCurrent,offset CmdBuf	;set current ptr to start of buffer
	mov	ax,word ptr scInfo.EntryCol ;set cursor to entry location
	mov	word ptr scInfo.CurrentCol,ax
	jmp	SyncCursor		;sync onscreen cursor to scInfo
GotoStartOfLine	endp

;-----------------------------------------------------------------------------

GotoEndOfLine	proc	near
	call	Get_cbCurrent_cbLast	;get cb pointers
	mov	cx,di			;cx= amount we need to forward
	sub	cx,si
	jmp	ForwardCxChars		;advance cursor
GotoEndOfLine	endp

;-----------------------------------------------------------------------------

BackSpace proc	near
	call	BackupChar		;back cursor up one
	.if	zero?			;if couldn't back up (left margin),
	  ret				;do nothing
	.endif
BackSpace endp

; NOTE: fall through to DeleteChar...

;-----------------------------------------------------------------------------
;
; Various deletion routines.

	.repeat				;dummy block (to support breaks)

DeleteChar proc	near
	  mov	cx,1
	  .break

DeleteToChar label near
	  xor	cx,cx			;assume no deletion will be done
	  call	GetKeybCharNoEcho	;char to delete to (but not including)
	  .break .if !zero?		;break= dual-byte key, ignore request
	  call	Get_cbCurrent_cbLast	;get cb pointers
	  lea	cx,[di+1]		;cx= onscreen chars past cursor + 1
	  sub	cx,si
	  mov	di,si			;scan for char to delete up to
	  repne scasb
	  jcxz	@F			;jmp= char not found, ignore request
	    lea	cx,[di-1]		;chars to delete
	    sub	cx,si
	  @@:
	  .break

DeleteBefore label near
	  mov	ax,cbCurrent		;ax= count of chars in front of cursor
	  sub	ax,offset CmdBuf
	  push	ax			;save for deletion
	  call	GotoStartOfLine		;move cursor to start of line
	  pop	cx			;cx= chars to delete (0-n)
	  .break

DeleteAll label near			;delete entire CmdBuf
	  call	GotoStartOfLine

DeleteAfter label near
	  mov	ch,1			;delete from current (no carry below!)

	.until	1			;end dummy block (to support breaks)

DeleteCxChars label near
	jcxz	@F			;jmp= nothing to delete
	  call	Get_cbCurrent_cbLast	;get cb pointers
	  push	word ptr [di]		;save onscreen text terminator
	  mov	byte ptr [di],0		;terminate onscreen text
	  mov	bx,di			;save end of onscreen text
	  mov	di,si			;di= start of onscreen text
	  add	si,cx			;si= start of text following deletion
	  .if	si > bx			;if deleting more than onscreen,
	    mov	si,bx			;limit for proper screen update
	  .endif
	  call	OverwriteFormattedMsg	;fixup display
	  pop	word ptr [bx]		;restore onscreen text terminator
	  sub	si,di			;si= number of onscreen chars deleted
	  sub	cbLast,si		;update end of onscreen chars ptr
	  mov	si,di			;si,di= cursor location
	  .repeat			;find char beyond deletion
	    .break .if byte ptr [si] == 0 ;stop if hit end of all text
	    inc	si
	  .untilcxz
	  .repeat			;remove deleted chars
	    lodsb
	    stosb
	  .until al == 0
	@@:
	ret
DeleteChar endp

;-----------------------------------------------------------------------------

BackupWord proc	near
	.repeat				;back up till non-whitespace
	  call	BackupChar
	  .break .if zero?		;break= at left margin, stop
	  mov	al,[si]			;al= character under cursor
	  call	IsWhitespace
	.until	!zero?

	.while	1			;now back up till prev is whitespace
	  .break .if si == offset CmdBuf ;break= hit left margin, stop
	  dec	si
	  lodsb
	  call	IsWhitespace
	  .break .if zero?
	  call	BackupChar
	.endw

	ret
BackupWord endp

;-----------------------------------------------------------------------------

ForwardWord proc near

	call	Get_cbCurrent_cbLast	;get cb pointers (need si to start)
	.while	1			;skip till whitespace
	  lodsb
	  .break .if al == 0		;break= at end-of-line, stop
	  call	IsWhitespace
	  .break .if zero?		;break= hit whitespace
	  call	ForwardChar
	.endw

	.repeat				;skip till non-whitespace
	  call	ForwardChar
	  .break .if zero?		;break= at end-of-line, stop
	  lodsb
	  call	IsWhitespace
	.until	!zero?

	ret
ForwardWord endp

;-----------------------------------------------------------------------------

CopyTemplateToChar proc	near
	call	GetKeybCharNoEcho	;get char to copy up to from template
	.if	zero?			;ignore if dual-byte keystroke,
	  call	Get_cbCurrent_cbLast	;get cb pointers
	  .repeat			;scan to matching character
	    cmp	byte ptr [si],1		;carry/non-zero if hit end of entry
	    .break .if carry?		;break= matching char not found
	    inc	si
	  .until [si] == al		;loop if char not found

	  .if	zero?			;if matching char found,
	    mov	bx,si			;save endpoint
	    .repeat			;advance cursor till matching char
	      call ForwardChar		;advance cursor
	    .until si == bx		;loop until endpoint hit
	  .endif
	.endif

	ret
CopyTemplateToChar endp

;-----------------------------------------------------------------------------
; Copy chars to CmdBuf till end of template

CopyToTemplateEnd proc near

	.repeat
	  call	ForwardChar
	.until	zero?

	ret
CopyToTemplateEnd	endp

;-----------------------------------------------------------------------------

CopyAllToTemplate proc near
	call	GotoStartOfLine
	mov	di,offset CmdBuf	;remove visible CmdBuf from display
	mov	si,offset SingleZero
	mov	cbLast,di		;(reset end of displayed text ptr)
	jmp	OverwriteFormattedMsg
CopyAllToTemplate endp

;-----------------------------------------------------------------------------

	.repeat				;dummy block (to support breaks)
PageDown  label	near
	  mov	ax,slHistory.Tail	;make last history row active
	  mov	slHistory.Active,ax

SelectPrevious label near
	  call	BackupActiveHistory	;back up to previous row (if any)
	  .break			;load selected row

PageUp	  label	near
	  mov	ax,slHistory.Head	;make first history row active
	  mov	slHistory.Active,ax
	  .break			;load selected row

SelectNext label near
	  .break .if cbLast == offset CmdBuf ;no advance if command line clear
	  call	AdvanceActiveHistory
	.until	1			;end dummy block (to support breaks)

; NOTE: fall through to LoadActiveHistory...

;-----------------------------------------------------------------------------

LoadActiveHistory proc near
	call	DeleteAll		;erase command text/reset cbCurrent
	call	GetActiveHistoryRowLength ;cx= active row length
	mov	si,slHistory.Active	;si= start of active history row
	mov	di,offset CmdBuf	;di= start of command buffer
	push	di			;save for use again below
	rep movsb			;move in selected text (may be empty)
	mov	[di],cl			;terminate CmdBuf
	mov	si,di			;set new cb pointers
	call	Set_cbCurrent_cbLast
	pop	si			;si= start of command buffer
	jmp	DisplayFormattedMsg	;display new command buffer contents
LoadActiveHistory endp

;-----------------------------------------------------------------------------

ExtendLineDown proc near
	mov	dx,offset AdvanceActiveHistory ;direction of movement

	.if	0
ExtendLineUp label near
	  mov	dx,offset BackupActiveHistory ;direction of movement
	.endif

	mov	si,cbCurrent		;si= address of current CmdBuf char
	push	[si]			;terminate CmdBuf (for searching)
	mov	byte ptr [si],0
	push	si

	mov	bx,offset slHistory	;bx= history string list structure
	assume	bx:near ptr STRINGLIST
	push	[bx].Active		;search startpoint (for quitting)
	.repeat				;loop to search history,

		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; Change active history row up/down.  If we hit the top or
		; bottom of the history list, wrap to the other end.
		;
		; Note the list may be empty, in which case the below code
		; will do nothing - that's OK, because the string will not
		; be matched below, and we'll exit after it doesn't.

	  mov	si,[bx].Active		;selection ptr prior to movement
	  call	dx			;select previous/next history row
	  .if	si == [bx].Active	;if no movement (at top or bottom),
	    mov	di,[bx].Head		;assume we'll wrap back to start
	    .if	si == di		;if that's where we began,
	      mov di,[bx].Tail		;wrap to bottom instead
	    .endif
	    mov	[bx].Active,di
	  .endif

		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; Match selected entry against portion of CmdBuf.  Note the
		; search routine searches the whole list from di, so if a
		; match is found, we need to insure it matched the first
		; (the selected) list entry.

	  mov	si,offset CmdBuf	;compare CmdBuf to next row
	  mov	di,[bx].Active
	  call	StringListPoolSearch
	  .if	zero?			;if we found a match,
	    cmp	si,[bx].Active		;may only be against selected entry
	  .endif
	  stc				;carry= assume match was made
	  .break .if zero?		;break= matched selected entry

	  pop	ax			;refresh search startpoint
	  push	ax
	.until	ax == [bx].Active	;loop until we've searched entire list
	pop	ax			;ax= search startpoint

	pop	si			;restore CmdBuf image
	pop	[si]

	.if	carry?			;if a match was made above,
	  sub	si,offset CmdBuf	;si= count of orignal prefix chars
	  push	si			;save for cursor reset below
	  call	LoadActiveHistory	;load selected history row
	  call	GotoStartOfLine		;put cursor at start of line
	  pop	cx			;number of chars to advance
	  jmp	ForwardCxChars		;put cursor to where it started out
	.endif
	mov	[bx].Active,ax		;nothing found, don't change active
	assume	bx:nothing

	ret
ExtendLineDown endp

;-----------------------------------------------------------------------------

EnterCommandNumber proc	near
	mov	ax,slHistory.Head	;ignore if history is empty
	cmp	slHistory.Tail,ax
	je	@F

	call	DeleteAll		;erase entire input
	mov	si,offset LinePrompt	;display prompt/insert into CmdBuf
	.repeat
	  lodsb
	  push	si
	  call	StoreChar
	  pop	si
	.until	si == offset LinePrompt + sizeof LinePrompt

	.while	1			;loop to obtain number,
	  call	GetKeybCharNoEcho
	  .continue .if !zero?		;ignore dual-byte keystroke chars
	  .if	al == ESCAPE		;if aborting command,
	    call DeleteAll		;empty command buffer
	    @@:
	    ret				;and exit
	  .endif
	  .break .if al == CR		;stop if CR
	  call	Get_cbCurrent_cbLast	;get cb pointers
	  .if	al == BS		;if backspace,
	    .if	di != offset CmdBuf + sizeof LinePrompt ;if char(s) to delete,
	      call BackSpace		;erase previous char
	    .endif
	    .continue
	  .endif
	  .if al >= '0' && al <= '9'	;if digit,
	    .if di != offset CmdBuf + sizeof LinePrompt + 5 ;if not at max,
	      call StoreChar		;append to buffer
	    .endif
	  .endif
	.endw

	xor	bx,bx			;init number accumulator
	mov	si,offset CmdBuf + sizeof LinePrompt ;start of entry
	.while	1			;loop to convert to binary
	  lodsb
	  sub	al,'0'			;convert next digit
	  .break .if carry?		;break= hit terminating 0
	  cbw				;zero extend
	  xchg	bx,ax			;add to accumulator
	  mul	Constant10
	  .if	!carry?			;if no overflow,
	    add	bx,ax			;cx= new accumulator
	  .endif
	  sbb	dx,dx			;if overflow, keep accumulator at max
	  or	bx,dx
	.endw

	mov	di,slHistory.Head	;select first history entry
	.repeat				;loop to advance to chosen entry,
	  mov	slHistory.Active,di
	  .break .if bx < 2		;break= advancing done
	  dec	bx			;reduce amount left to advance
	  call	GetActiveHistoryRowLength ;chack active history entry
	.until	di == slHistory.Tail	;loop until no more history entries
	jmp	LoadActiveHistory	;load active history row

EnterCommandNumber endp

;-----------------------------------------------------------------------------

DisplayMacros proc near
	mov	al,0			;indicate macro display

	skip
DisplayHistory label near
	  mov	al,1			;indicate history display
	endskip

	cbw				;extend & save macro/history flag
	push	ax

	call	DeleteAll		;remove all input from screen

	pop	cx			;restore macro/history flag

	mov	ax,slMacros.Head	;assume macro display
	mov	di,offset NumberBuffer + 7
	mov	byte ptr [di],ch
	jcxz	@F			;if assumption wrong,
	  mov	ax,slHistory.Head	;load history address
	@@:
	xchg	bx,ax			;bx= selected string list address

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Ready to display string list.  si= address of selected string list,
	; di= address of work string in NumberBuffer (initialized to null
	; string), cx= 1 (if numbering, this will be the row count), else 0.

	mov	scInfo.CurrentRow,0ffh	;force cursor to 0,0 after next CRLF
	.while	byte ptr [bx] != 0	;while more strings to display,

	  jcxz	@F			;if numbering (history list),
	    mov	di,offset NumberBuffer + 5 ;format end of "nn>:" string
	    mov	ax,'>:'			;assume selected row
	    .if bx != slHistory.Active	;if assumption wrong,
	      mov ah,' '		;correct it
	    .endif
	    mov  word ptr [di],ax
	    mov  ax,cx
	    .repeat			;format in digits
	      xor  dx,dx
	      div  Constant10
	      dec  di
	      add  dl,'0'
	      mov  byte ptr [di],dl
	    .until ax == 0
	    inc  cx			;bump row number for next pass
	  @@:
		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; Display the next string virtually.  If it won't fit
		; onscreen, issue a "-- More --" message before displaying it
		; physically.
		;
		; Note it might have been easier simply to put the -- More --
		; logic in the lowest-level display routine; however, that
		; would lead to multi-row objects potentially being broken up
		; by a "more" message, and I didn't want to let that happen.

	  mov	ax,word ptr scInfo.CurrentCol ;ax= current cursor location
	  .if	ah != 0ffh		;if not top, check if --more-- needed,
	    push ax			;save current cursor location
	    call DisplayCRLFVirtual	;virtual display CRLF
	    mov  si,di			;virtual display number, if numbering
	    call DisplayFormattedMsgVirtual
	    mov  si,bx			;virtual display next string entry
	    call DisplayFormattedMsgVirtual
	    mov  al,scInfo.CurrentRow	;al= post-virtual-display row number
	    pop  word ptr scInfo.CurrentCol ;restore current cursor location
	    inc  ax			;set to TotalRows if we've hit bottom
	    .if  al == scInfo.TotalRows	;if time for -- More -- prompt,
	      mov  si,offset MoreMsg	;display -- More --
	      call DisplayFormattedMsg
	      call GetKeybCharNoEcho	;wait for keystroke
	      .break .if ax == ESCAPE	;break= ESC entered, abort
	      mov  scInfo.CurrentRow,0ffh ;force cursor to 0,0 after next CRLF
	    .endif
	  .endif

	  call	DisplayCRLF		;get to start of next line
	  mov	si,di			;display number, if numbering
	  call	DisplayFormattedMsg
	  mov	si,bx			;display next formatted string,
	  call	DisplayFormattedMsg
	  mov	bx,si			;advance to next list entry
	.endw

	jmp	ExitKeystrokeDispatcher ;exit (force return of empty line)
DisplayMacros endp

;-----------------------------------------------------------------------------
;
; End Module
;
;-----------------------------------------------------------------------------

	ENDSEG
	end	start
