	page	,132

;-----------------------------------------------------------------------------
;
;  This file is part of doskey.com.
; 
;  Copyright (C) 2001-2011 Paul Houle (http://paulhoule.com)
; 
;  This program is free software; you can redistribute it and/or modify
;  it under the terms of the GNU General Public License as published by
;  the Free Software Foundation; either version 2 of the License, or
;  (at your option) any later version.
; 
;  This program is distributed in the hope that it will be useful,
;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;  GNU General Public License for more details.
; 
;  You should have received a copy of the GNU General Public License
;  along with this program; if not, write to the Free Software
;  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
;
;  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;
;		Enhanced DOSKEY
;
; Source for enhanced DOSKEY.
;
; Change history (just started 12/12/2001).
;  20011212:	caused completions to occur in lowercase.
;		fixed bug with forward arrow going beyond end of buffer.
;  20030121:	dkres: fixed bug allowing tab completions to extend command
;		line beyond end of buffer.
;  20030121:	dktab: remove trailing backslash if tab completion session
;		ends w/a CR  (for easier use with CD command).
;  20031022:	allowed chars 7f-ff to be part of filename/path.  This fixes
;		problems with umlauts.
;  201407/08:	Major enhancements such as LFN support and new CHANGE mode.
; --------------
; Stuff that might be helpful to do:
;
; *)	Further implement Win95 extensions? (QueryClose already implemented)
; *)	Search MACROS for command completion first?
; *)	Search internal commands during command completion?
;	(complicated; also need to consider cases when 4DOS for example is
;	used as the command shell; not sure if it is really useful anyway)
; *)	Sorting of names in alphabetic order for auto-completion?
;	(most useful probably when the user has not typed any partial names;
;	however, there are efficiency issues for sorting algorithms)
; *)	Simple compression of string lists? (macros, history, help msg, etc)?
;	(however, the actual file size and memory footprint may be enlarged
;	with the addition of the code for string lists compression)
; *)	DOSKEY @macro=file	(define multi-line macro)
;	DOSKEY @=file		(define immediate exec macro, perhaps use
;				 this to implement startup multi-macro def).
;	(Note that $T can already separate commands, not sure if above useful)
; *)	Move new macro def to macro buf manually, then make all macro ops
;	near (es==ds)?
; *)	Whole screen display subsystem is a mess - redesign.
;
; Functional differences from Microsoft DOSKEY:
;
;  (1)	No DOS version check, except for the availability of specific
;	features (e.g. support for quotation marks even when LFN API is not
;	present) only available in later versions of DOS.
;  (2)	Closed standard handles (STDOUT, STDIO, etc) before TSR installation.
;	This was done to prepare for (3), since to unload I simply release
;	the TSR memory block - this would leave the standard handles open.
;  (3)	Implemented '/UNINSTALL' feature/switch that unloads DOSKEY from
;	memory.  Not all that useful in practice, but doesn't cost anything
;	in installed space, and sure is handy during development (so I don't
;	need to reboot to test the next program change).
;  (4)	Command line options can begin with '-' as well as '/'.
;  (5)	No bells (in error messages, or during command line overflow).
;  (6)	ALTF8 works as F8, only forward instead of reverse.
;  (7)	F10 displays macros.
;  (8)	Commands "macroname=xxx" have the same functionality as
;	"doskey macroname=xxx" once DOSKEY is installed, but it is shorter
;	 and will not re-load the executable and thus also faster.
;  (9)	Implemented function key=macro logic (allow func key redefinition).
;	Valid function keys are from F1 to F12. A macro name with the name
;	of the function key plus a "!" sign (e.g. "f5!") will cause the value
;	of the macro to be immediately executed after the corresponding
;	function key is pressed.
; (10)	ESCAPE resets command history pointer to bottom of list.
; (11)	Tab/Ctrl+tab/Shift+tab completion with full LFN support.
; (12)	ALTF6 generates a tab character, ALTF5 toggles LFN support, ALTF9
;	toggles between append mode and change mode, and ALTF11 toggles the
;	inclusion of system and hidden files.
;
;-----------------------------------------------------------------------------

	.nolist
	include	dkdefs.inc	;include constant definitions, externdef's
	.list
	option	noljmp		;disallow automatic jmp-lengthening

;-----------------------------------------------------------------------------
;
; Define entry point, which immediately jumps to the exec handling code at
; the end of the image.
;
; Note the "near ptr" is needed to get the correct "1st pass" size of this
; jump (so the sjmpend offset is correct).

	org	100h		;org to define entry jmp
start	label	near
	jmp	near ptr Startup ;jump to exec code at end of image
sjmpend	equ	$

;-----------------------------------------------------------------------------
;
;	<<<<<<<<  Start of installed variable declarations  >>>>>>>>
;
; *** NOTE: changing/adding/moving/redefining variables in ANY WAY	***
; *** necessitates the modification of DOSKEY_ID; otherwise, an attempt	***
; *** to use the new DOSKEY command to update/interrogate an installed	***
; *** TSR would be fatal.						***
;
; We utilize the PSP to hold run-time buffers and variables, and include it
; as part of the instance data (this keeps us from otherwise wasting this
; portion of the installed image).  Fields declared in the PSP cannot be
; initialized at compile-time, and cannot be used at startup time (by the
; exec code), since the PSP fields they overlay (incoming command line,
; environment address, etc.) need to be accessed during startup.
;

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Start of uninitialized fields that overlay the first 128 bytes of
	; the PSP.  These fields should be used by the resident code only.

	org	0
SwiData0Start label	near	;start of first instance data block

R2SIZE	equ	sizeof PSP.pspReserved2 ;(masm complains unless this is equ'd)
	if	R2SIZE gt sizeof ENTRYSTACKFRAME
	  .err	<EntryFrame won't fit in PSP.pspReserved2>
	endif
	org	PSP.pspReserved2
EntryFrame ENTRYSTACKFRAME <>	;entry int 2fh stack frame moved here

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Overlay run-time data areas with (unused) PSP FCBs.
	;
	; The int 2fh address should be kept fixed at offset PSP.pspFCB_1.
	; This is critical for development only - we can unload a modified
	; version (with a new code and/or data layout) with it fixed;
	; otherwise, the int 2fh vector restore could be fatal.

	org	PSP.pspFCB_1
Old2fVector dd	?		;pre-installation int 2f vector
slMacros  STRINGLIST <>		;macro string list control structure
slHistory STRINGLIST <>		;command history list control structure
scInfo	  SCREENINFO <>		;screen information

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Command buffer control fields.  When editing is underway, cbCurrent
	; is the address of the cursor location, and cbLast is the address
	; beyond the last Template byte that has been made officially part
	; of CmdBuf via an editing key (like F1).
	; cbActive is not used during editing.  After editing is complete,
	; it is pointed to the start of the command, which is executed
	; from there.  If on re-entry cbActive points to a non-zero (the
	; prior command didn't fully execute, or was a macro), cbActive is
	; assumed to point within CmdBuf, and CmdBuf execution is continued
	; from that point (note: if cbActive does point to a zero, that zero
	; byte may not be within CmdBuf).

cbCurrent dw	?		;current (cursor) address within CmdBuf
cbLast	  dw	?		;CmdBuf highwater offset of Template
cbActive  dw	?		;points to non-zero when CmdBuf macro "active"

FCBend	  equ	$
IOBufSize dw	?

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; CmdBuf fields are placed at offset PSP.pspCommandTail.  This
	; is as if it were an incoming command line (for no real reason,
	; except that it makes things a bit easier to debug).
	;
	; Note the extra CmdBuf byte (129 total instead of 128) - it could be
	; accessed during temporary termination, so leave it there.
	;
	; The CmdBufLen is not used during command input.

	org	PSP.pspCommandTail
	if	$ lt FCBend
	  .err	<pspFCB overlay area too large>
	endif
CmdBufLen	db	?	;CmdBuf len prefix *** MUST PRECEDE CmdBuf ***
CmdBuf		db 255 dup (?)	;command input buffer
CmdBufLimit	db	0	;CmdBuf limit (last possible terminator slot)
CmdLineLimit	dw	?	;actual command line limit (CmdBuf + 128~255 - 1)
cmode		db	?	;for several options
tflags		db	?	;run-time flags
	if	$ lt sjmpend
	  .err	<sjmpend/tflags mismatch>
	endif

;-----------------------------------------------------------------------------

NumberBuffer	db	8 dup(0)
PrevChar	db	0

SwiData0End	label	near	;#############################################

Constant10	dw	10		;for use by mul/div opcodes

MoreMsg		db	CR,'-- More --',0

	;non-filename/path characters >20h and <7fh (i.e., in printable range)
SeparatorList	db	'"/<>|'		;characters usually considered as separators
					;N.B. wildcards are not separators
LFNallowable	db	'+[]'		;characters allowed in LFN and macro names
					;they are generally not allowed in SFN
CommandPrefix	db	' ,=;'		;characters allowed to prefix command
					;these are the same set of characters only
					;allowed in LFN + must be quoted
PathEquals	db	'path=',0	;used to scan environment for path
CommandExts	db	'.com',0	;recognized command extension table
		db	'.bat',0
		db	'.exe',0
		db	0		;(terminate CommandExts table)
DirCmds		db	'cd',0
		db	'chdir',0
		db	'md',0
		db	'mkdir',0
		db	'rd',0
		db	'rmdir',0
		db	0
DollarSrc db	'tlgb$',CMDSEP,0	;must be in descending binary value
DollarSub db	CMDSEP,'<>|$',CMDSEP,0	;characters to substitute for above

StarDotStar	db	'*.*'		;used to terminate find first masks
SingleZero	db	0		;single zero/StarDotStar terminator

LinePrompt	db	'Line number: '	;used for history row selection (F9)

MacrosFullMsg	db	'Insufficient memory for macro. Reinstall DOSKEY with a larger /BUFSIZE.',CR,0

;-----------------------------------------------------------------------------
; The below mess defines the keystroke dispatching tables.  All the macro
; nonsense is simply to allow the pairing of the keystroke and handler
; address via DKEY invocations.

DKEY	macro	a,a1,b,b1,c,c1,d,d1,e,e1,f,f1,g,g1,h,h1
	ifnb	<a>
	  DKEY2	%(a),a1
	  DKEY	b,b1,c,c1,d,d1,e,e1,f,f1,g,g1,h,h1
	endif
	endm
DKEY2	macro	keystroke,dispatch
	ifb	KeyString
KeyString textequ <db >,<keystroke>
	else
KeyString catstr KeyString,<,keystroke>
	endif
	dw	dispatch
	endm

tablejmp label	word
KeyString textequ <>
 DKEY	CR,FinishLineAndExit,	LF,DoNothing  ;021018, CTL_F,DoNothing ;021018
 DKEY	ESCAPE,ClearCmdBuf,	BS,DBackSpace
 DKEY	TAB,TabComplete,
 DKEY	5,GotoEndOfLine,	1,GotoStartOfLine	;basic "emacs" control
 DKEY	2,DBackupChar,		6,DForwardChar		;sequences added for
 DKEY	11,DeleteAfter,		4,DDeleteChar		;  Scott Baker 021018
t1str	textequ KeyString
KeyString textequ <>
 DKEY	0,DStoreChar,		CTRLTAB,TabComplete,	F12,F12KEY
 DKEY	F1,F1KEY,		RIGHT,DForwardChar,	INSKEY,ToggleInsert
 DKEY	LEFT,DBackupChar,	F6,F6KEY,		HOME,GotoStartOfLine
 DKEY	DEL,DDeleteChar,	CTLEND,DeleteAfter,	CTLHOME,DeleteBefore
 DKEY	CTLLEFT,BackupWord,	CTLRIGHT,ForwardWord,	F4,F4KEY
 DKEY	F2,F2KEY,		F5,F5KEY,		F3,F3KEY
 DKEY	ENDKEY,GotoEndOfLine,	DOWN,SelectNext,	UP,SelectPrevious
 DKEY	PAGEUP,PageUp,		PAGEDOWN,PageDown,	F7,F7KEY
 DKEY	F10,F10KEY,		ALTF10,ClearMacros,	ALTF7,ClearHistory
 DKEY	F8,F8KEY,		ALTF8,ExtendLineDown,	F9,F9KEY
 DKEY	ALTF5,ToggleLFN,	ALTF6,GenerateTab,	F11,F11KEY
 DKEY	ALTF9,ToggleACMode,	ALTF11,ToggleSysHidden
 dw	DoNothing		;*** default dual-byte handler
table1	t1str
table2	KeyString
table2end label byte		;end of dual-byte table

;-----------------------------------------------------------------------------

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Task switcher startup info structure, and instance data block
	; descriptor list.  The instance data block list contains contiguous
	; descriptors that describe blocks of memory that a task switcher
	; (e.g., Windows) will keep unique between multiple DOS sessions.
	;
	; We have two such blocks of memory:
	;
	;   1) Generic variables, from SwiData0Start to SwiData0End.
	;   2) Block of memory that will hold macros and history.
	;
	; The portions of the instance data block descriptor structures that
	; cannot be filled in at assembly time are initialized during init.

SwiData0 SWINSTANCEITEM <offset SwiData0Start, SwiData0End - SwiData0Start>
SwiData1 SWINSTANCEITEM <offset EndResident, DEFAULT_BUFSIZE>
	 dd	0		;terminate SWINSTANCEITEM list

				;switcher startup info structure
SwStartup SWSTARTUPINFO <3, 0, 0, 0, offset SwiData0>

;-----------------------------------------------------------------------------
;
; Get command line, either input by user, or expanded from a continuing
; command line or macro execution.
;
; IN:
;  interrupts are enabled
;  EntryFrame.TemplatePtr= address of incoming command line buffer.  First
;	byte is the incoming buffer size, which has been verified externally
;	to be IOBufSize (128-255).  This byte is not examined or changed
;	here.
;
;	The 2nd byte holds the size of the incoming command buffer, which is
;	used as the editing template (e.g., for the F1 key).  Up to a max of
;	255 characters are copied from the incoming buffer to our CmdBuf,
;	for use as the template.
;
;	On exit, the 2nd byte will receive the returned input size, not
;	counting the CR attached to the end of the input.  This will be zero
;	if a macro is entered - the actual macro command(s) will be returned
;	in subsequent call(s).
;
;	The third byte and beyond hold the actual input: there will be a max
;	of IOBufSize - 1 input bytes returned, followed by a single CR
;	character.
; OUT:
;	Template[0]= unchanged (never examined - see above).
;	Template[1]= cmd length, not including terminating CR.
;		If a macro is entered, this will be zero.  The next input
;		call will return the first command line for the macro.
;	Template[2]= first command line character - CR if returned line empty.

GetCmdLine	proc	near

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; First check if we're executing a macro.  If so, continue its
	; execution.

	mov	si,slMacros.Active	;si= current active macro address
	mov	cx,IOBufSize	;cx= max chars that can be returned
	dec	cx
	lea	di,CmdBuf
	add	di,cx
	mov	CmdLineLimit,di
	.if	byte ptr [si] != 0	;if a macro is executing,
	  les	di,EntryFrame.TemplatePtr ;point beyond returned buffer sizes
	  inc	di
	  inc	di

	  .while 1			;loop to process macro characters,
	    lodsb			;get next macro char
	    .if	al == '$'		;if $x translation needed,

	      lodsb			;get next byte
	      .if al <= '9'		;if may be a parameter transfer,
		.if al == '*' || al > '0' ;if valid parameter index,
		  call TransferMacroParameter ;transfer the parameter
		  .continue		;continue macro execution
	        .elseif al == '0'
		  call TransferMacroName ;transfer the name also
		  .continue
		.endif
	      .endif
	      
	      call ToLower		;convert to lowercase
	      mov bx,offset DollarSrc - 1 ;descending list of $ pairings, 0
	      .repeat			;scan for mapped pairing
		inc bx			;next valid pairing
	      .until al >= [bx]
	      .continue .if !zero?	;continue= invalid pairing, ignore
	      mov al,(DollarSub - DollarSrc)[bx] ;map to substitution char
	    .endif

	    .break .if al == 0		;break= end of macro, exit
	    .break .if al == CMDSEP	;break= command separator, exit

	    jcxz @F			;ignore if output buffer is full
	      stosb			;transfer macro char to command
	      call DisplayFormattedAl	;(display chars as we transfer them)
	      dec cx			;reduce size of output buffer
	    @@:
	  .endw

	  .if	al == 0			;if hit macro terminator above
	    dec  si			;don't go beyond macro terminator
	    mov  bx,cbActive		;discard command used for argument(s)
	    .repeat
	      mov al,[bx]
	      .break .if al == 0	;don't go beyond end of command
	      inc bx
	    .until al == CMDSEP
	    mov  cbActive,bx
	  .endif
	  mov	slMacros.Active,si	;update active macro pointer
	  jmp short TerminateCommand
	.endif

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Macro not executing - either continue execution of a Ctl<T>
	; separated command line (if one is pending), or prompt user for a
	; new entry.

	mov	si,cbActive		;CmdBuf execution pointer
	mov	bx,offset DisplayFormattedAl ;assume echo as we transfer
	.if	byte ptr [si] == 0	;if CmdBuf inactive (not executing),
	  call	EditCmdBuf		;get new user entry
	  mov	si,offset CmdBuf	;see if new entry a multi-part command
	  .repeat
	    lodsb
	    cmp al,CMDSEP		;multi-part command?
	    jz  TerminateEmpty		;jmp= yes, execute it like a macro
	  .until al == 0
	  mov si,offset CmdBuf		;reset ptr to beginning of command
	  mov bx,offset DoNothing	;no re-display of just-entered command
	.endif

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	;090612 Kludge to disable QueryClose (Win 9x-only) call if running
	;090612 on DOS/Windows that doesn't support the int 2fh function.
	;
	;090629 Added logic to also disable QueryClose if Windows GUI is not
	;090629 running.
	;201408 Rewrote the entire detection code for better compatibility.

	;mov ah,30h			;090612 *** START
	;int 21h				;get OS version
	;or	di,si			;zero if Windows API not present
	push ax
	push bx
	mov ax,160ah			;check Windows version
	int 2fh
	mov di,offset QueryClose	;cause QueryClose to return non-zero
	.if ax || bh < 4		;if not at least Win95 (Win4+) running
	  mov ax,0ff0ch			;  write "or al,0ffh" opcode
	  stosw
	  mov al,0c3h			;  write "retn" opcode
	.else				;restore opcodes if under Win9x
	  mov ax,0ba52h
	  stosw
	  mov al,0
	.endif				;090612 *** END
	stosb
	pop bx
	pop ax

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; If a macro is found at the start of the command, 'activate' the
	; macro (put us in "macro execution" state).  This causes an empty
	; command line to be returned for this call.  On the next call, the
	; first command spawned by the macro will be returned.

	call	CheckMacroDef
	.if al & GF_EXIT
		call	ActivateMacro	;try to parse-off/activate macro
	.endif
	jz	TerminateEmpty		;jmp= execute macro next time in

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Execute next command from CmdBuf.

	mov	cx,IOBufSize	;cx= max chars that can be returned
	dec	cx
	les	di,EntryFrame.TemplatePtr ;point to return buffer
	inc	di
	inc	di
	.while	1
	  mov	al,[si]			;stop if we've hit terminator
	  .break .if al == 0
	  inc	si			;bypass character
	  .break .if al == CMDSEP	;stop if command separator hit
	  jcxz @F			;ignore if output buffer is full
	    stosb			;transfer macro char to command
	    call bx			;(display chars, unless inhibited)
	    dec cx			;reduce size of output buffer
	  @@:
	.endw
	mov	cbActive,si		;update CmdBuf execution pointer
	jmp short TerminateCommand	;jmp= terminate command and exit

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Here to terminate with an empty command line.

TerminateEmpty:

	les	di,EntryFrame.TemplatePtr ;exit, returning null command
	inc	di
	inc	di

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Here to terminate a command and exit.

TerminateCommand:

	mov	al,CR			;terminate generated command
	stosb
	mov	ax,word ptr EntryFrame.TemplatePtr
	xchg	ax,di			;ax= length of command (without CR)
	inc	di
	sub	ax,di
	sub	al,2
	stosb				;return length of generated command

	ret

CheckMacroDef:
	push si
	mov bp,1
	call ParseMacroDef
	dec bp
	pop si
	push ax
	.if !(al & GF_EXIT)
	  call ProcessMacro
	  inc scInfo.EntryRow
	  call DisplayCRLF
	  .if cx && bl
	    push si
	    mov si,di
	    call DisplayFormattedMsg
	    xor cx,cx
	    mov dx,-1
	    call BackupCxChars
	    call DisplayCRLF
	    mov ah,scInfo.CurrentRow
	    inc ah
	    .if ah >= scInfo.TotalRows
	      sub ah,3
	      .if ah > scInfo.EntryRow
	        push ax
	        call DisplayCRLF
	        pop ax
	      .endif
	      inc ah
	      inc ah
	      mov scInfo.EntryRow,ah
	    .endif
	    pop si
	  .else
	    call EndLastLines
	    mov scInfo.EntryRow,ah
	  .endif
	  call DeleteAll
	.endif
	pop ax
	ret

ProcessMacro label near
	call	GetMacroDefinitionInfo	;get macro definition info
	.if	!zero?			;if a definition/deletion present,
	  call DeleteMacro		;delete if matching entry found
	.endif
	.if cx			;cx=0 if no definition, or deletion only
	  call StoreMacro		;store macro definition
	.endif
	ret

;-----------------------------------------------------------------------------
; Gets command-line macro definition information.
;
; IN:
;   ds= local segment
;   es= unused
; OUT:
;   si= addr of start of parsed macro definition string (if any)
;   di= addr of character beyond '=' within macro definition string
;   cx= entire macro definition size, if a macro is being defined;
;	zero if no macro definition or macro deletion only ("macro=")
;   non-zero status if macro definition (or deletion) was parsed

GetMacroDefinitionInfo proc near

	mov	si,offset CmdBuf	;si= start of parsed macro definition
	xor	cx,cx
	.if	[si] != cl		;if macro alteration parsed,
	  push	si			;find '=' in macro definition
	  .repeat
	    lodsb
	  .until al == '='
	  mov	di,si			;save address immediately beyond '='
	  pop	si
	  .if [di] != cl		;if not simply deleting macro,
	    mov cl,CmdBufLen		;cx= macro definition length
	    inc cx
	  .endif
	  or	di,di			;set non-zero (flag definition parsed)
	.endif
	ret
GetMacroDefinitionInfo endp

;-----------------------------------------------------------------------------
; Delete installed TSR macro, if a matching one exists.  This should only
; be called if a 'macro=xxx' (xxx may be empty) definition was found on the
; command line.
;
; We only delete the stored macro if there will be enough room
; to write the new definition in its place (assuming a "new
; definition" exists, which is indicated by a non-zero cx).
;
; We attempt to keep the Active pointer correct; but, if the active
; entry is deleted, Active is pointed to the string list terminator.
;
; IN:
;   si/cx= definition start/length (as set byGetMacroDefinitionInfo)
;   di= beyond macro '=' separator (as set byGetMacroDefinitionInfo)
;   es= installed TSR segment
; OUT:
;   si/cx= unchanged
;   di= destroyed
;   bx= amount of potential macro freespace following deletion

DeleteMacro proc near
	push	si			;save macro definition info
	push	cx

	mov	bx,es:slHistory.Next	;bx= history freespace that could
	sub	bx,es:slHistory.Head	;  be used to define macro)
	sub	bx,HISTORY_MIN		;(reserve minimum history space)
	.if	carry?			;(should never be below HISTORY_MIN,
	  xor	bx,bx			;but we are, no history space to use
	.endif

	push	cx			;save macro definition length
	push	word ptr [di]		;save definition area we'll modify
	push	di			;save so we undo the modification
	mov	byte ptr [di],0		;terminate definition following '='
	mov	di,es:slMacros.Head	;es:di= TSR macro string list pool
	call	StringListPoolSearch	;search for macro name in string pool
	pop	di			;undo modification of macro definition
	pop	word ptr [di]
	pop	cx			;restore macro definition length
	.if	zero?			;if macro found, delete definition,
	  push	ds			;set up to modify installed TSR memory
	  push	es
	  pop	ds
	  mov	di,si			;scan past end of entry to delete
	  xor	ax,ax
	  .repeat
	    inc	bx			;(update potential freespace)
	    scasb
	  .until zero?
	  .if	cx <= bx		;only if redefined macro would fit,
	    mov	ax,slMacros.Active	;prepare to maintain Active
	    .if	ax < di			;if deletion affects Active,
	      add ax,di			;assume Active entry not deleted
	      sub ax,si
	      .if ax >= di		;if assumption wrong,
		mov ax,slMacros.Tail	;reset to string list terminator
	      .endif
	      mov slMacros.Active,ax	;store modified active pointer
	    .endif
	    mov cx,si			;cx= bytes to pack to higher memory
	    sub cx,slMacros.Head
	    std				;move reverse
	    cmpsb			;point at first byte to move
	    rep movsb			;pack previous data to higher memory
	    cld
	    inc di			;di= point to last byte moved
	    mov slMacros.Head,di		;shrink macro string list pool
	    mov slHistory.Next,di	;enlarge history string list pool
	  .endif
	  pop	ds
	.endif

	pop	cx			;restore macro definition info
	pop	si
	ret
DeleteMacro endp

;-----------------------------------------------------------------------------
; Store installed TSR macro.  This should only be called if a 'macro=xxx'
; (xxx must NOT be empty) definition was found on the command line.
;
; We only store the macro if enough room to write the definition
; is available; if not, we issue an error.
;
; The Active pointers are maintained for both the macro and history
; string lists (the macro pointer will never be lost; the history pointer
; can be if its active entry is deleted).
;
; IN:
;   si/cx= definition start/length (as set by GetMacroDefinitionInfo)
;   es= installed TSR segment
;   bx= amount of macro freespace (set by DeleteMacro).
; OUT:
;   si/cx= unchanged
;   bl= 0 if storage successful, else GF_EXIT or GF_ERROR;
;	if bl non-zero, di holds error message address.

StoreMacro proc near

	cmp	bx,cx			;carry if not enough macro room
	mov	di,offset MacrosFullMsg ;assume macros full error
	mov	bl,GF_EXIT or GF_ERROR
	.if	!carry?			;if room to store macro,
	  push	ds			;set up to modify installed TSR memory
	  push	es
	  pop	ds
	  assume bx:near ptr STRINGLIST
	  call	GetHistoryFreespace	;get history string list freespace

	  sub	[bx].Next,cx		;take freespace from history
	  mov	bx,offset slMacros	;enlarge macro area
	  sub	[bx].Head,cx
	  sub	[bx].Tail,cx
	  sub	[bx].Active,cx
	  mov	di,[bx].Head
	  mov	si,di
	  add	si,cx
	  mov	cx,[bx].Tail
	  inc	cx			;(move terminator too)
	  sub	cx,[bx].Head
	  rep movsb

	  pop	ds			;store macro definition

	  call	GetMacroDefinitionInfo	;get macro definition info
	  call	StringListAppend	;store macro definition

	  xor	bx,bx			;indicate macro storage successful
	.endif

	ret
StoreMacro endp

GetCmdLine	endp

;-----------------------------------------------------------------------------
; Parse macro definition.
;
; IN:
;   ax= parsed option flags
;   si= current 0-terminated command line position within CmdBuf
; OUT:
;   ax= parsed option flags, (GF_EXIT or GF_ERROR) set if error.
;   si= advanced beyond parsed, formatted macro definition
;   di= (if error) error message text.

ParseMacroDef proc near
	push	ax			;save parsed option flags

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Parse and format macro definition string (if one supplied on
	; command line).  The formatted macro string is placed at the
	; start of CmdBuf.
	;
	; The Macro name is converted to lowercase, whitespace is removed on
	; either side of the '=' separating the name from the definition,
	; and the string is zero-terminated.  Note whitespace at the end of
	; the definition is retained.
	; The macro length  (w/o the trailing 0) is stored in CmdBufLen.

	.repeat				;start dummy block (to support breaks)
	  .if bp == 1
	    call	SkipSpaceTab	;skip whitespace before macro name
	    push	si
	    mov		di,offset CmdBuf;put formatted macro definition here
	    .repeat			;loop to transfer macro name
	      lodsb			;get next suspected name char
	      inc	di
	      call	IsMacroNameChar	;zero if al is a macro name character
	    .until !zero?		;loop if we stored a valid name char
	    pop		si
	    .if di == offset CmdBuf + 1 || al != '='	;break= no name found, al= terminator
	      mov al,GF_EXIT
	      .break
	    .endif
	  .else
	    call SkipWhitespace		;skip whitespace before macro name
	  .endif
	  mov	di,offset CmdBuf	;put formatted macro definition here
	  .repeat			;loop to transfer macro name
	    lodsb			;get next suspected name char
	    call ToLower		;convert to lower case
	    stosb			;  and store
	    call IsMacroNameChar	;zero if al is a macro name character
	  .until !zero?			;loop if we stored a valid name char
	  dec	si			;point to macro name terminator
	  dec	di
	  mov	cx,di			;cx= macro name characters parsed
	  sub	cx,offset CmdBuf
	  .break .if zero?		;break= no name found, al= terminator
	  call SkipWhitespace		;skip whitespace before '='
	  xor	al,'='			;see if next character is '='
	  .break .if !zero?		;break= macro definition error
	  movsb				;move the '='
	  
	  .repeat			;we don't include consecutive = signs right after the
	    lodsb			;macro name as part of the text; but if a user really
	  .until	al != '='	;wants to add one or more = signs as the leading part
	  dec	si			;of the text, add any number of spaces/tabs beforehand
	  call SkipSpaceTab		;skip leading spaces or tabs
	  .repeat			;move text following '='
	    inc	cx			;increase macro length
	    lodsb
	    stosb
	  .until al == 0		;up to terminator
	.until	1			;end dummy block
	mov	CmdBufLen,cl		;store macro size (w/o terminator)

	cmp	al,0			;non-zero if error parsing definition
	pop	ax			;restore parsed option flags
	.if	!zero?			;if definition error,
	  mov	di,offset InvalidMacroDefMsg	;cause exit with message
	  mov	al,GF_EXIT or GF_ERROR		;  and non-zero errorlevel
	.endif
	ret
ParseMacroDef endp

;-----------------------------------------------------------------------------
; Here to substitue a $0-$9 or $* macro parameter.
;
; IN:
;   al= parameter indicator (*,0-9)
;   cx= remaining output buffer space
;   es:di= output buffer pointer

TransferMacroName:
	push si
	mov si,offset CmdBuf
	call SkipSpaceTab
	.repeat
		lodsb
		call IsWhitespace
		.break .if zero? || al == 0 || al == CMDSEP
		call IsMacroNameChar
		.break .if !zero? && al != '[' || al > 39h && al < 40h || al == '"' || al == ',' || al == '/'
		stosb
		call DisplayFormattedAl
		dec cx
	.until 0
	pop si
	ret

TransferMacroParameter proc near

	sub	al,'*'			;assume '*'
	.if	!zero?			;if assumption wrong,
	  sub	al,'0' - '*'		;convert to 1-9
	.endif
	cbw
	xchg	bx,ax

	push	si			;save current macro address
	mov	si,cbActive		;si= command buffer parameters

	.while	1			;loop to find selected parameter,
	  call	SkipSpaceTab		;skip whitespace in front of parameter
	  dec	bx			;break= selected parameter found
	  jle @F;.break .if sign? || zero?
	  .repeat			;skip to start of next parameter
	    mov	al,[si]			;get suspected parameter byte
	    .break .if al == 0 || al == CMDSEP ;don't skip beyond command end
	    inc	si			;advance beyond parameter byte
	    call IsWhitespace
	  .until zero?			;loop until end of parameter
	.endw
	@@:

	.while	1			;loop to transfer parameter
	  lodsb				;get next parameter byte
	  .break .if al == 0 || al == CMDSEP ;break= end of command line, exit
	  or	bx,bx			;transferring entire line?
	  .if	!sign?			;if not,
	    call IsWhitespace
	    .break .if zero?		;break= end of parameter, exit
	  .endif
	  jcxz	@F			;ignore if output buffer is full
	    stosb			;transfer macro char to command
	    call DisplayFormattedAl	;(display chars as we transfer them)
	    dec	cx			;reduce size of output buffer
	  @@:
	.endw

	pop	si			;restore current macro address

	ret
TransferMacroParameter endp

;-----------------------------------------------------------------------------
;
; Here we edit the command line on the screen, and add it to the history.

EditCmdBuf	proc	near

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; The first thing we do is initialize the SCREENINFO structure
	; (screen/cursor size, entry/current cursor location).
	;
	; This information is needed for repositioning the cursor during
	; deletion, -- more -- messages, toggling insert mode, etc.
	;
	; First try to get the number of screen rows from ansi.sys.  This
	; requires a DISPLAYMODE buffer - get scratch space for it from
	; the history string list pool.
	;
	; Note: we use the STDERR device in the expectation that there's
	; less chance that it's been redirected.

	mov	cx,sizeof DISPLAYMODE	;di= get freespace for DISPLAYMODE
	call	GetHistoryFreespace
	assume	di:near ptr DISPLAYMODE
	mov	ax,14			;(size of fields after dmDataLength)
	mov	[di].dmInfoLevel,ah	;init DISPLAYMODE for get info call
	mov	[di].dmDataLength,ax

	mov	dx,di			;get ansi.sys char device info (all
	mov	cx,37fh			;  we're interested in is the number
	mov	bx,STDERR		;  of screen rows)
	mov	ax,440ch
	int	21h
	mov	ax,[di].dmRows		;assume success, load number of rows

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; If ansi.sys failed, or we're in graphics mode, try to get number
	; of screen rows from the system BIOS.  If this looks wrong, just use
	; 25.

	.if	carry? || [di].dmMode != 1 ;if error, or not in text mode,
	  mov	al,25			;assume 25 (can't get from BIOS)
	  xor	cx,cx			;fetch BIOS video rows
	  push	ds
	  mov	ds,cx
	  mov	cl,ds:VIDEO_ROWS	;get video row count - 1
	  pop	ds
	  jcxz	@F			;jmp= value unreliable
	    xchg ax,cx			;value may be good, use it
	    inc	 ax
	  @@:
	.endif
	assume	di:nothing
	mov	scInfo.TotalRows,al

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Get number of screen columns and incoming cursor location and mode
	; from the system BIOS.

	mov	ah,0fh			;ah= number of screen columns
	int	10h			;bh= current video page
	mov	scInfo.TotalCols,ah
	mov	ah,3			;get current cursor location
	int	10h
	xchg	ax,dx
	mov	word ptr scInfo.EntryCol,ax ;ah= entry row, al= entry column
	mov	word ptr scInfo.CurrentCol,ax
	mov	scInfo.EntryMode,cx	;ch= cursor start, cl= cursor end

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Initialize the flags.
	

	mov	al,tflags		;al= flags
	and	al,not (TF_INSERT or TF_EXIT)	;kill insert, exit request
	.if	al & TF_STARTINSERT	;if insert needs to be on
	  or	al,TF_INSERT		;turn it on
	.endif
	mov	tflags,al		;save initial flags

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Set the current insert mode, and corresponding cursor size.

	call	InitInsertMode		;set cursor mode appropriately

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Init CmdBuf to template contents, and edit it.

	call	CopyTemplateToCmdBuf
	.repeat
	  call	GetKeybCharNoEcho
	  call	DispatchKeystroke

	  call	QueryClose		;close DOS box request sensed?
	  .if	zero?			;if yes,
	    call DeleteAll		;erase entire line,
	    .break			;  and exit keystroke entry loop
	  .endif

	.until	tflags & TF_EXIT	;loop until exit requested

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Restore entry cursor mode.

	mov	cx,scInfo.EntryMode	;restore entry cursor mode
	mov	ah,1
	int	10h

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Discard unused template characters (terminate transferred chars
	; with a zero), and add command to history if it is non-empty.

SaveHistory label near
	mov	si,offset CmdBuf	;si= start of user input
	mov	cbActive,si		;point to command to execute
	mov	di,cbLast		;di= end of user input
	mov	byte ptr [di],0		;discard unused template characters
	.if	si != di		;if entry non-empty,
	  assume bx:near ptr STRINGLIST
	  mov	bx,offset slHistory	;store entry to command history
	  call	StringListAppend
	  mov	ax,[bx].Tail		;set selected past bottom of history
	  mov	[bx].Active,ax
	  assume bx:nothing
	.endif
	ret

EditCmdBuf	endp

;-----------------------------------------------------------------------------

ToggleInsert	label	near
	xor	tflags,TF_INSERT	;invert current insert state

InitInsertMode proc	near

	mov	cx,scInfo.EntryMode	;cx= cursor mode on entry
	mov	al,tflags		;al= current flags

	push	ds			;prepare to access BIOS shift status
	xor	bx,bx
	mov	ds,bx
	mov	bx,BIOS_SHIFT_STATUS
	and	byte ptr [bx],not SS_INSERT ;assume insert off
	.if	al & TF_INSERT		;if assumption wrong,
	  sub	ch,2			;set insert on (widen cursor)
	  .if	carry?
	    mov	ch,0
	  .endif
	  or	byte ptr [bx],SS_INSERT
	.endif
	pop	ds

	mov	ah,1		; Video display   ah=functn 01h
	int	10h		;  set cursor mode in cx

	ret
InitInsertMode endp

;-----------------------------------------------------------------------------
;
; ax = keystroke.
; ah= 0 for ascii al, ah= 1 for dual-byte (extended) code in al.
;
; No other registers altered.
; 
; Returns non-zero if character is a dual-byte keystroke (ah= 1).

KeybProc proc	near private

	.repeat
			; No keystroke available, allow other system operation
	  int	28h			;allow TSR operation (keyboard idle)
	  mov	ax,1680h		;MS-DOS idle call (same as int 28h)
	  int	2fh

GetKeybCharNoEcho label	near
	  call	QueryClose		;close DOS box request sensed?
	  mov	ax,ESCAPE		;assume yes, set up ESCAPE keystroke
	  jz	@F			;jmp= exit if close DOS box sensed

	  mov	ah,0bh			; DOS Services  ah=function 0Bh
	  int	21h			;  if keybd char available,al=FF

	.until	al != 0
	mov	ah,8			; DOS Services  ah=function 08h
	int	21h			;  get keybd char al, no echo
	@@:

	and	ax,0ffh			;assume single byte character
	.if	zero?			;if assumption wrong
	  mov	ah,8			;get 2nd byte
	  int	21h
	  cmp	al,0			;shouldn't happen, but would be fatal
	  jz	GetKeybCharNoEcho	;jmp= guarantee NEVER return al == 0
	  mov	ah,1			;indicate dual-byte code
	.endif

	or	ah,ah			;non-zero if dual-byte keystroke
	ret
KeybProc endp

;-----------------------------------------------------------------------------
; Test for external DOS box close (Windows '95 extension).
;
; IN:
;   none
; OUT:
;   zero status if DOS box close requested
;   ax destroyed; all other registers preserved

QueryClose proc	near
	push	dx
	mov	dx,0100h		;Win '95 extension: query close
	mov	ax,168Fh
	int	2fh			;check for external close request
	pop	dx
	test	ax,ax			;zero status if close request sensed
	ret
QueryClose endp

;-----------------------------------------------------------------------------
; IN:
;   si= start of potential macro name
;
; OUT:
;   non-zero status if macro not matched:
;	cbActive= unchanged
;	slMacros.Active= unchanged
;
;   zero status if macro was matched and activated:
;	cbActive= set to beyond macro name in source (si)
;	slMacros.Active= set to definition of macro (beyond =)
;
;   si, bx= preserved

ActivateMacro	proc	near
	push	si
	call	CheckMacro
	.if zero?
	  mov	cbActive,si		;save address beyond matched name
	  mov	slMacros.Active,di	;save address of macro definition text
	.endif
	pop	si
	ret

ActivateMacro	endp

CheckMacro:
	call	SkipSpaceTab
	.repeat				;dummy block (to support breaks)
	  mov	di,si			;save start of potential macro name
	  lodsb				;zero if 1st char valid as macro name
	  call	IsMacroNameChar
	  .break .if !zero?		;break= no macro found

	  .repeat			;scan to end of macro name
	    lodsb
	    call IsMacroNameChar
	  .until !zero?
	  dec	si			;point to name terminator

	  push	word ptr [si]		;terminate name (for searching)
	  push	si
	  mov	word ptr [si],'='
	  mov	si,di			;search for macro definition
	  mov	di,slMacros.Head
	  call	StringListPoolSearch
	  pop	si			;undo early termination
	  pop	word ptr [si]
	.until	1			;end dummy block (to support breaks)
	ret

;-----------------------------------------------------------------------------
; IsMacroNameChar:
;	Sets zero status if al is a valid macro name character.
;
; IsPathChar:
;	Sets zero status if al is a valid file/path character.
;
; IsWhitespace:
;	Sets zero status if al is whitespace.
;
; al= unchanged

	.repeat
	  test	al,al			;not macro name char - ret w/nz status
	  ret
IsMacroNameChar	label	near

	  .if	al != ':' && al != '?' && al != '*'	;zero status if not a macro name char
	    cmp	al,'\'
	  .endif
	.until	!zero?
	call IsLFNQuoteChar
	.if !zero?
	  ret
	.endif

IsPathChar proc near		;zero status if al is a file/path character allowed in LFN
	push	di
	.if al == '|'		;check for the special case of the pipe char ("|")
	  inc si		;it is valid as the second byte of a DBCS Character,
	  call CheckSBChar	;but otherwise will not be valid as a path char
	  dec si
	  .if di != -1
	    pop di
	    cmp al,al
	    ret
	  .endif
	.endif
	push	cx
	mov	cx,sizeof SeparatorList + 1 ;cx= separator count + 1
;031022	.if al > 20h && al < 7fh	;if printable, may be file/path char,
	.if al >= 20h			;only allow chars >=20h,		;031022
	  mov	di,offset SeparatorList	;list of separator chars
	  repne scasb			;cx= 0 if al not found in list
	.endif
	test	cx,cx			;zero status if file/path character
	pop	cx
	pop	di
	ret
IsPathChar endp

IsLFNPathChar proc near		;zero status if al is not a character only allowed in LFN but not SFN
	push	di
	push	cx
	mov	cx,sizeof LFNallowable + 1
	mov	di,offset LFNallowable
	repne	scasb
	test	cx,cx
	pushf
	.if !zero? && al != '+'	;"[" and "]" are valid as the second byte of a DBCS Character
	  inc si		;even in SFN, but otherwise will only be valid if used in LFN
	  call CheckSBChar
	  dec si
	  .if di != -1
	    popf
	    cmp al,al
	  .else
	    popf
	  .endif
	.else
	  popf
	.endif
	pop	cx
	pop	di
	ret
IsLFNPathChar endp

IsLFNQuoteChar proc near	;LFNQuoteChars are same as command prefix characters
	push	di		;When these characters appear in file/path name, they must be quoted
	push	cx
	mov	cx,sizeof CommandPrefix + 1
	mov	di,offset CommandPrefix
	repne	scasb
	test	cx,cx
	pop	cx
	pop	di
	ret
IsLFNQuoteChar endp

IsWhitespace	proc	near
	.if	al != ' '
	  cmp	al,TAB
	.endif
	ret
IsWhitespace	endp

;-----------------------------------------------------------------------------
;
; Skips whitespace at ds:si.  si is left pointing at 1st non-whitespace.
;
;  al is set to character si is left pointing to.
;  all other registers saved.

SkipWhitespace proc	near

	.repeat
	  lodsb
	.until	al != ' ' && al != '"' && al != TAB
	dec	si

	ret
SkipWhitespace	endp

SkipSpaceTab proc	near

	.repeat
	  lodsb
	.until	al != ' ' && al != TAB
	dec	si

	ret
SkipSpaceTab	endp

;-----------------------------------------------------------------------------
;
; Convert al to lowercase

ToLower	proc	near
	.if	al >= 'A' && al <= 'Z'
	  add	al,'a' - 'A'
	.endif
	ret
ToLower	endp

;-----------------------------------------------------------------------------
; Clear all History.
;
; Clear all macros.
;
; Clear history is trivial.  Clear macros requires we shrink the macro
; string list pool to 1 byte (and appropriately enlarge the history string
; pool).

ClearMacros proc	near
	assume	bx:near ptr STRINGLIST
	mov	bx,offset slMacros
	mov	ax,[bx].Next		;ax= last macro area byte
	dec	ax
	mov	[bx].Head,ax		;shrink macro area to 1 byte
	mov	slHistory.Next,ax	;enlarge history region appropriately
	.if	0
ClearHistory label near
	  mov	bx,offset slHistory
	.endif
	jmp	StringListClear
	assume	bx:nothing
ClearMacros endp

;-----------------------------------------------------------------------------
; si, bx, dx= preserved

BackupActiveHistory	proc	near
	mov	di,slHistory.Active	;di= selected history line
	mov	cx,di			;cx= history bytes that precede it
	sub	cx,slHistory.Head
	std				;scan to start of previous entry
	dec	di
	dec	di
	mov	al,0
	repne scasb			;(note: may read 1 byte BEFORE start -
	inc	di			;  that's OK, it's in our segment)
	inc	di
	cld
	mov	slHistory.Active,di	;save new selected entry
	ret
BackupActiveHistory	endp

;-----------------------------------------------------------------------------
; cx= length of active history row w/o zero terminator (0 if at end of list)
; di= points beyond terminator
; si, bx, dx= preserved

GetActiveHistoryRowLength proc near
	mov	di,slHistory.Active	;di= selected history line
	mov	cx,-1			;scan past end of entry
	mov	al,0
	repne scasb
	inc	cx			;cx= count of chars, w/o terminator
	not	cx
	ret
GetActiveHistoryRowLength endp

;-----------------------------------------------------------------------------
; si, bx, dx= preserved

AdvanceActiveHistory	proc	near
	call	GetActiveHistoryRowLength ;cx= active history row length
	jcxz	@F			;jmp= no active row (at end of list)
	  mov	slHistory.Active,di	;advance active beyond current row
	@@:
	ret
AdvanceActiveHistory	endp

;-----------------------------------------------------------------------------

ClearCmdBuf label near
	mov	ax,slHistory.Tail	;reset history to bottom of list
	mov	slHistory.Active,ax	;(DOSKEY doesn't do this, but should)

	call	DeleteAll		;erase entire line

; NOTE: fall through to CopyTemplateToCmdBuf...

;-----------------------------------------------------------------------------
;
; Resets command buffer to contents of incoming template.  Also initializes
; the CmdBuf control variables (cbxxx).

CopyTemplateToCmdBuf	proc	near
	mov	si,offset CmdBuf
	mov	di,si
	call	Set_cbCurrent_cbLast	;set new cb pointers
	push	ds
	lds	si,EntryFrame.TemplatePtr
	xor	ax,ax
	inc	si
	lodsb
	.if	ax >= IOBufSize
	  mov	ax,IOBufSize
	  dec	ax
	.endif
	xchg	cx,ax

	jcxz	@F
	  .repeat
	    lodsb
	    .break .if al == 0	;(0's would be fatal)
	    .break .if al == CR
	    stosb
	  .untilcxz
	@@:
	pop	ds
	xor	ax,ax		;indicate not in macro execution state
	stosb			;terminate CmdBuf
CopyTemplateToCmdBuf	endp

; NOTE: fall through to cbActiveDisable...

;-----------------------------------------------------------------------------
; Disables any "active" command by pointing cbActive to a zero.  Note the zero
; we point to is NOT within CmdBuf - the fact that *cbActive is zero is all
; that is required.

cbActiveDisable	proc	near
	mov	cbActive,offset SingleZero
	ret
cbActiveDisable	endp

;-----------------------------------------------------------------------------

DispatchKeystroke proc	near

	lea	si,PrevChar
	mov	di,offset table1	;assume single byte character
	mov	cx,(table2 - table1) + 1
	.if	ah != 0			;if assumption wrong,
	  mov	di,offset table2 + 1	;use dual-byte character table
	  mov	cl,(table2end - (table2 + 1)) + 1
	.elseif byte ptr [si] && al >= 40h
	  mov ah,[si]
	  mov byte ptr [si],0
	  call Get_cbCurrent_cbLast
	  mov di,si
	  inc di
	  push ax
	  mov ah,-1
	  call CheckDBCSChar
	  pop ax
	  .if si > di || tflags & TF_INSERT && ah == -1
	    dec di
	    push di
	    push ax
	    call StoreCharInsertMode
	    pop ax
	    call Get_cbCurrent_cbLast
	    pop di
	    .if si == di
	      push ax
	      call BackSpace
	      pop ax
	      .if !(tflags & TF_INSERT)
	        mov al,ah
	        call StoreCharInsertMode
	      .endif
	    .endif
	  .else
	    call StoreChar
	  .endif
	  ret
	.endif
	mov byte ptr [si],0
	repne scasb			;scan selected table
	sub	di,offset table1 + 1	;di= jump table index
	add	di,di
	jmp	tablejmp[di]

DispatchKeystroke endp

;-----------------------------------------------------------------------------
; gets	si= cbCurrent
;	di= cbLast
;
; no other registers altered

Get_cbCurrent_cbLast	proc	near	;load cb Current/Last pointers
	mov	si,cbCurrent
	mov	di,cbLast
	ret
Get_cbCurrent_cbLast	endp

;-----------------------------------------------------------------------------
; sets	cbCurrent= si
;	cbLast= di
;
; If si > di, di set to si
;
; no other registers altered

Set_cbCurrent_cbLast proc near	;store cb Current/Last pointers
	.if	si > di		;advance template inclusion ptr, if needed
	  mov	di,si
	.endif
	mov	cbCurrent,si
	mov	cbLast,di
	ret
Set_cbCurrent_cbLast	endp

;-----------------------------------------------------------------------------

FinishLineAndExit proc	near		;handles entry of a CR
	call	Get_cbCurrent_cbLast	;get cb pointers
	mov	byte ptr [di],0		;terminate entry (truncate template)
	call	DisplayFormattedMsg	;display to move to end of entry
;extra linefeed generated, is the CR needed???
;???	mov	al,CR			;move to left margin
;???	call	DisplayFormattedAl

ExitKeystrokeDispatcher label near
	or	tflags,TF_EXIT		;force keystroke dispatcher exit

DoNothing label	near
	ret
FinishLineAndExit endp

CheckFunKeyDef:				;handler for function key redefinition
	lea si,NumberBuffer
	push [si]
	push [si+2]
	push si
	mov byte ptr [si],'f'
	inc si
	.if ah < 0ah
	  add ah,30h
	.else
	  mov byte ptr [si],'1'
	  inc si
	  add ah,26h
	.endif
	mov [si],ah
	inc si
	mov byte ptr [si],'!'
	mov di,si
	inc si
	mov byte ptr [si],0
	pop si
	push si
	push di
	push ax
	call CheckMacro
	pop ax
	pop di
	pop si
	.if zero?
	  push si
	  call DeleteAll
	  pop si
	  push si
	  call StoreStringInsertMode
	  pop si
	  pop [si+2]
	  pop [si]
	  jmp short FinishLineAndExit
	.endif
	mov byte ptr [di],0
	push si
	push ax
	call CheckMacro
	pop ax
	pop si
	.if zero?
	  push si
	  call DeleteAll
	  pop si
	  push si
	  call StoreStringInsertMode
	  mov al,' '
	  call StoreCharInsertMode
	  pop si
	  xor ah,ah
	.endif
	pop [si+2]
	pop [si]
	ret

F1KEY:
	mov ah,1
	call CheckFunKeyDef
	.if ah
	  call DForwardChar
	.endif
	ret
F2KEY:
	mov ah,2
	call CheckFunKeyDef
	.if ah
	  call CopyTemplateToChar
	.endif
	ret
F3KEY:
	mov ah,3
	call CheckFunKeyDef
	.if ah
	  call CopyToTemplateEnd
	.endif
	ret
F4KEY:
	mov ah,4
	call CheckFunKeyDef
	.if ah
	  call DeleteToChar
	.endif
	ret
F5KEY:
	mov ah,5
	call CheckFunKeyDef
	.if ah
	  call CopyAllToTemplate
	.endif
	ret
F6KEY:
	mov ah,6
	call CheckFunKeyDef
	.if ah
	  call GenerateCtlZ
	.endif
	ret
F7KEY:
	mov ah,7
	call CheckFunKeyDef
	.if ah
	  call DisplayHistory
	.endif
	ret
F8KEY:
	mov ah,8
	call CheckFunKeyDef
	.if ah
	  call ExtendLineUp
	.endif
	ret
F9KEY:
	mov ah,9
	call CheckFunKeyDef
	.if ah
	  call EnterCommandNumber
	.endif
	ret
F10KEY:
	mov ah,0ah
	call CheckFunKeyDef
	.if ah
	  call DisplayMacros
	.endif
	ret
F11KEY:
	mov ah,0bh
	call CheckFunKeyDef
	.if ah
	  call DisplayCompletion
	.endif
	ret
F12KEY:
	mov ah,0ch
	call CheckFunKeyDef
	ret

;-----------------------------------------------------------------------------

GenerateTab label near
	mov	al,TAB
	skip
GenerateCtlZ label near
	  mov	al,CTL_Z
	endskip

; NOTE: fall through to DStoreChar...

;-----------------------------------------------------------------------------
; DStoreChar - stores char with DBCS support on DBCS-enabled systems.
; StoreChar - stores char in al according to current insert mode.
; StoreCharInsertMode - inserts char in al, regardless of current insert mode.
; StoreStringInsertMode - inserts asciiz string at si (regardless of
;	insert mode).  Char(s) that won't fit are ignored.
;
; bx preserved, all other regs destroyed

DStoreChar:
	push ax
	lea si,NumberBuffer
	mov di,si
	inc di
	push [si]
	mov [si],al
	mov byte ptr [si+1],40h
	mov ah,-1
	call CheckDBCSChar
	pop [si]
	pop ax
	.if tflags & TF_INSERT
	  .if si > di
	    call Get_cbCurrent_cbLast
	    mov di,si
	    push di
	    call StoreChar
	    call Get_cbCurrent_cbLast
	    pop di
	    .if si > di
	      lea di,PrevChar		;if we inserted a DBCS leading byte successfully,
	      mov byte ptr [di],-1	;mark it so that if the insertion of the second
	    .endif			;byte failed (no space left), remove it too
	    ret
	  .else
	    jmp short StoreChar
	  .endif
	.endif
	.if si > di			;check if we are overwriting a character with
	  call Get_cbCurrent_cbLast	;DBCS leading byte; if true, proceed normally
	  mov di,si			;unless the current byte is a single-byte char
	  inc di			;and the next byte is another DBCS leading byte.
	  push ax			;In such case we have to replace the single-byte
	  mov al,[di-1]			;character with a double-byte one so that the
	  mov ah,-1			;other DBCS leading byte will not be replaced by
	  call CheckDBCSChar		;the second byte of the current DBCS character
	  .if al && si == di
	    lea di,PrevChar
	    mov [di],al
	  .endif
	  pop ax
	  jmp short StoreChar
	.endif
	call	Get_cbCurrent_cbLast	;get cb pointers
	.if byte ptr [si] && byte ptr [si+1]
	  inc si
	  inc si
	  call CheckSBChar		;check for case that we need to overwrite a double-
	  .if di != -1			;byte character consisting of two bytes by a regular
	    push ax			;single-byte character on systems that support DBCS;
	    call DDeleteChar		;if yes, delete the double-byte char and then insert
	    pop ax
	    jmp short StoreCharInsertMode
	  .endif
	.endif
StoreChar proc	near
	mov	cl,tflags		;get flags (for insert state)
	skip
StoreCharInsertMode label near
	  mov	cl,TF_INSERT		;force insert mode
	endskip
	mov	si,offset SingleZero	;only store/insert the char in al
	.if	0
StoreStringInsertMode label near
	  lodsb				;al= 1st byte to insert
	  mov	cl,TF_INSERT		;force insert mode
	.endif

	push	bx

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; First update CmdBuf by inserting/overwriting the new data.  The
	; screen will be updated later.

	mov	bx,si			;fetch additional source chars via bx
	call	Get_cbCurrent_cbLast	;get cb pointers
	mov	dx,CmdLineLimit		;limit for terminator
	and	cx,TF_INSERT		;cx= non-zero if inserting
;030121	push	si			;save entry cursor location
	.while	al != 0			;while more char(s) to store,
	  jcxz	@F			;if inserting,
	    .break .if di == dx		;break= buffer full, stop inserting
	    push si
	    .repeat			;insert new char into buffer
	      xchg [si],al
	      inc  si
	    .until al == 0 || si == dx	;until terminator, or buffer limit
	    mov	[si],ch			;re-terminate
	    pop	si
	    inc di			;advance high water mark	;030121
	  .if	0			;else overwriting,
	    @@:
	    .break .if si == dx		;break= buffer full, stop overwriting
	    mov	ah,[si]			;load char being overwritten
	    mov	[si],al			;overwrite char
	    .if	ah == 0			;if extending buffer,
	      mov [si+1],ah		;re-terminate
	    .endif
	  .endif
	  inc	si			;advance storage ptr
	  mov	al,[bx]			;fetch next char to store
	  inc	bx
	.endw
	mov	bx,si			;bx= end of insertion
;030121	pop	si			;restore entry cursor location
	call	Get_cbCurrent_cbLast	;reload original cb pointers	;030121

	sub	bx,si			;bx= count of stored chars (0-n)
	jz	StoreCharRet

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; If there is something onscreen beyond the current cursor position
	; (i.e., we're not simply extending the command line), we have to
	; adjust the display accordingly - we can't simply display the new
	; chars, even if overwriting, since the display widths may differ.

	push	bx			;save count of stored char(s)
	.if	si != di		;if we updated screen beyond cursor,
	  mov	al,ah			;assume we didn't insert
	  dec	bx
	  jcxz	@F			;jmp= assumption correct
	    inc	bx			;undo assumption, restore insert size
	    add	di,bx			;insertion always advances cbLast
	    mov	cbLast,di
	    mov	al,[si+bx]		;set up for OverwriteFormattedMsgAl
	  @@:
	  push	word ptr [di]		;save char beyond end of display
	  push	di
	  mov	[di],ch			;terminate displayed text
	  lea	di,[si+bx]		;update display (cursor and beyond)
	  call	OverwriteFormattedMsgAl
	  pop	di			;restore char beyond end of display
	  pop	word ptr [di]
	.endif
	pop	cx			;cx= count of stored char(s)
	call	ForwardCxChars		;jmp= advance cursor

StoreCharRet:
	pop	bx
	ret
StoreChar endp

;-----------------------------------------------------------------------------
;
; Advances current cursor location 1 (or cx) CmdBuf char(s).  Advancement
; stops if end-of-buffer is hit.
;
; OUT:
;   si= current cursor address (advanced by chars we moved forward)
;   di= last onscreen address (potentially modified by advancement)
;   cx(ForwardCxChars only)= count of chars to move forward
;   ax/cx= destroyed
;   dx= count of chars forwarded (0-n)
;   all other registers preserved
;
;   zero status if no change to cursor location occurred

DForwardChar:
	call	Get_cbCurrent_cbLast	;get cb pointers
	inc	si
	inc	si
	call	CheckSBChar		;check for case that we need to backs
	.if di != -1			;up a double byte char on systems that
	  call ForwardChar		;support DBCS
	.endif	
ForwardChar proc near
	mov	cx,1			;forward 1 char
ForwardCxChars label near
	call	Get_cbCurrent_cbLast	;get cb pointers
	xor	dx,dx			;init count of amount forwarded
	jcxz	@F			;jmp= nothing to advance
	  .repeat			;for all chars to advance,

;011212	    lodsb			;get char to advance over/display
;011212	    .if al			;if not at end of entry/template
	    .if byte ptr [si] != 0	;if not at end of entry/template,
	      lodsb			;get char to advance over/display

	      call DisplayFormattedAl	;display char, advance cursor
	      inc  dx			;indicate advance was performed
	    .endif
	  .untilcxz
	@@:
	call	Set_cbCurrent_cbLast	;set new cb pointers
	or	dx,dx			;set return flags
	ret
ForwardChar endp

;-----------------------------------------------------------------------------
;
; Backs up current cursor location 1 or more chars.
;
; OUT:
;   si= current cursor address (either unchanged, or backed-up by 1)
;   zero status if no change to cursor location occurred.

DBackupChar:
	call	Get_cbCurrent_cbLast	;get cb pointers
	call	CheckSBChar		;check for case that we need to backs
	.if di != -1			;up a double byte char on systems that
	  call BackupChar		;support DBCS
	.endif	
BackupChar proc near
	mov	cx,1			;back up 1 char
BackupCxChars label near
	call	Get_cbCurrent_cbLast	;get cb pointers

	mov	ax,offset CmdBuf	;(can't back up before here)
	mov	di,si			;di= potential new cbCurrent, after
	sub	di,cx			;  attempting to back up cx chars
	.if	carry? || di < ax	;if backed up in front of left margin,
	  xchg	di,ax			;limit to left margin
	.endif
	.if	dl == -1 || di != si		;if backing up at least 1 char,
	  push	word ptr [di]		;save char under cursor, and location
	  push	di
	  mov	byte ptr [di],0		;terminate at new cursor location
	  .if dl == -1
	    mov al,scInfo.CurrentRow
	    .if al > scInfo.EntryRow
	      add dl,al
	      sub dl,scInfo.EntryRow
	    .endif
	    mov scInfo.EntryRow,al
	  .endif
	  mov	ax,word ptr scInfo.EntryCol ;set cursor to entry location
	  mov	word ptr scInfo.CurrentCol,ax
	  mov	si,offset CmdBuf	;display preceding chars virtually
	  call	DisplayFormattedMsgVirtual
	  call	SyncCursor		;sync onscreen/scInfo cursor location
	  pop	si			;restore character under cursor
	  pop	word ptr [si]
	  mov	cbCurrent,si		;save new current CmdBuf pointer
	  or	al,0ffh			;indicate cursor location was moved
	.endif

	ret
BackupChar endp

;-----------------------------------------------------------------------------

GotoStartOfLine	proc	near
	mov	cbCurrent,offset CmdBuf	;set current ptr to start of buffer
	mov	ax,word ptr scInfo.EntryCol ;set cursor to entry location
	mov	word ptr scInfo.CurrentCol,ax
	jmp	SyncCursor		;sync onscreen cursor to scInfo
GotoStartOfLine	endp

;-----------------------------------------------------------------------------

GotoEndOfLine	proc	near
	call	Get_cbCurrent_cbLast	;get cb pointers
	mov	cx,di			;cx= amount we need to forward
	sub	cx,si
	jmp	ForwardCxChars		;advance cursor
GotoEndOfLine	endp

;-----------------------------------------------------------------------------

DDeleteChar:
	call	Get_cbCurrent_cbLast	;get cb pointers
	inc si
	inc si
	call	CheckSBChar		;check for case that we need to delete
	.if di != -1			;a double byte char on systems that
	  call DeleteChar		;support DBCS
	.endif
	jmp DeleteChar
DBackSpace:
	call	Get_cbCurrent_cbLast	;get cb pointers
	call	CheckSBChar		;check for case that we need to remove
	.if di != -1			;a double byte char on systems that
	  call BackSpace		;support DBCS
	.endif	
BackSpace proc	near
	call	BackupChar		;back cursor up one
	.if	zero?			;if couldn't back up (left margin),
	  ret				;do nothing
	.endif
BackSpace endp

; NOTE: fall through to DeleteChar...

;-----------------------------------------------------------------------------
;
; Various deletion routines.

	.repeat				;dummy block (to support breaks)

DeleteChar proc	near
	  mov	cx,1
	  .break

DeleteToChar label near
	  xor	cx,cx			;assume no deletion will be done
	  call	GetKeybCharNoEcho	;char to delete to (but not including)
	  .break .if !zero?		;break= dual-byte key, ignore request
	  call	Get_cbCurrent_cbLast	;get cb pointers
	  lea	cx,[di+1]		;cx= onscreen chars past cursor + 1
	  sub	cx,si
	  mov	di,si			;scan for char to delete up to
	  repne scasb
	  jcxz	@F			;jmp= char not found, ignore request
	    lea	cx,[di-1]		;chars to delete
	    sub	cx,si
	  @@:
	  .break

DeleteBefore label near
	  mov	ax,cbCurrent		;ax= count of chars in front of cursor
	  sub	ax,offset CmdBuf
	  push	ax			;save for deletion
	  call	GotoStartOfLine		;move cursor to start of line
	  pop	cx			;cx= chars to delete (0-n)
	  .break

DeleteAll label near			;delete entire CmdBuf
	  call	GotoStartOfLine

DeleteAfter label near
	  mov	ch,1			;delete from current (no carry below!)

	.until	1			;end dummy block (to support breaks)

DeleteCxChars label near
	jcxz	@F			;jmp= nothing to delete
	  call	Get_cbCurrent_cbLast	;get cb pointers
	  push	word ptr [di]		;save onscreen text terminator
	  mov	byte ptr [di],0		;terminate onscreen text
	  mov	bx,di			;save end of onscreen text
	  mov	di,si			;di= start of onscreen text
	  add	si,cx			;si= start of text following deletion
	  .if	si > bx			;if deleting more than onscreen,
	    mov	si,bx			;limit for proper screen update
	  .endif
	  .if dl != -1
	    call	OverwriteFormattedMsg	;fixup display
	  .endif
	  xor dl,dl
	  pop	word ptr [bx]		;restore onscreen text terminator
	  sub	si,di			;si= number of onscreen chars deleted
	  sub	cbLast,si		;update end of onscreen chars ptr
	  mov	si,di			;si,di= cursor location
	  .repeat			;find char beyond deletion
	    .break .if byte ptr [si] == 0 ;stop if hit end of all text
	    inc	si
	  .untilcxz
	  .repeat			;remove deleted chars
	    lodsb
	    stosb
	  .until al == 0
	@@:
	ret
DeleteChar endp

;-----------------------------------------------------------------------------

BackupWord proc	near
	.repeat				;back up till non-whitespace
	  call	BackupChar
	  .break .if zero?		;break= at left margin, stop
	  mov	al,[si]			;al= character under cursor
	  call	IsWhitespace
	.until	!zero?

	.while	1			;now back up till prev is whitespace
	  .break .if si == offset CmdBuf ;break= hit left margin, stop
	  dec	si
	  lodsb
	  call	IsWhitespace
	  .break .if zero?
	  call	BackupChar
	.endw

	ret
BackupWord endp

;-----------------------------------------------------------------------------

ForwardWord proc near

	call	Get_cbCurrent_cbLast	;get cb pointers (need si to start)
	.while	1			;skip till whitespace
	  lodsb
	  .break .if al == 0		;break= at end-of-line, stop
	  call	IsWhitespace
	  .break .if zero?		;break= hit whitespace
	  call	ForwardChar
	.endw

	.repeat				;skip till non-whitespace
	  call	ForwardChar
	  .break .if zero?		;break= at end-of-line, stop
	  lodsb
	  call	IsWhitespace
	.until	!zero?

	ret
ForwardWord endp

;-----------------------------------------------------------------------------

CopyTemplateToChar proc	near
	call	GetKeybCharNoEcho	;get char to copy up to from template
	.if	zero?			;ignore if dual-byte keystroke,
	  call	Get_cbCurrent_cbLast	;get cb pointers
	  .repeat			;scan to matching character
	    cmp	byte ptr [si],1		;carry/non-zero if hit end of entry
	    .break .if carry?		;break= matching char not found
	    inc	si
	  .until [si] == al		;loop if char not found

	  .if	zero?			;if matching char found,
	    mov	bx,si			;save endpoint
	    .repeat			;advance cursor till matching char
	      call ForwardChar		;advance cursor
	    .until si == bx		;loop until endpoint hit
	  .endif
	.endif

	ret
CopyTemplateToChar endp

;-----------------------------------------------------------------------------
; Copy chars to CmdBuf till end of template

CopyToTemplateEnd proc near

	.repeat
	  call	ForwardChar
	.until	zero?

	ret
CopyToTemplateEnd	endp

;-----------------------------------------------------------------------------

CopyAllToTemplate proc near
	call	GotoStartOfLine
	mov	di,offset CmdBuf	;remove visible CmdBuf from display
	mov	si,offset SingleZero
	mov	cbLast,di		;(reset end of displayed text ptr)
	jmp	OverwriteFormattedMsg
CopyAllToTemplate endp

;-----------------------------------------------------------------------------

	.repeat				;dummy block (to support breaks)
PageDown  label	near
	  mov	ax,slHistory.Tail	;make last history row active
	  mov	slHistory.Active,ax

SelectPrevious label near
	  call	BackupActiveHistory	;back up to previous row (if any)
	  .break			;load selected row

PageUp	  label	near
	  mov	ax,slHistory.Head	;make first history row active
	  mov	slHistory.Active,ax
	  .break			;load selected row

SelectNext label near
	  .break .if cbLast == offset CmdBuf ;no advance if command line clear
	  call	AdvanceActiveHistory
	.until	1			;end dummy block (to support breaks)

; NOTE: fall through to LoadActiveHistory...

;-----------------------------------------------------------------------------

LoadActiveHistory proc near
	call	DeleteAll		;erase command text/reset cbCurrent
	call	GetActiveHistoryRowLength ;cx= active row length
	mov	si,slHistory.Active	;si= start of active history row
	mov	di,offset CmdBuf	;di= start of command buffer
	push	di			;save for use again below
	rep movsb			;move in selected text (may be empty)
	mov	[di],cl			;terminate CmdBuf
	mov	si,di			;set new cb pointers
	call	Set_cbCurrent_cbLast
	pop	si			;si= start of command buffer
	jmp	DisplayFormattedMsg	;display new command buffer contents
LoadActiveHistory endp

;-----------------------------------------------------------------------------

ExtendLineDown proc near
	mov	dx,offset AdvanceActiveHistory ;direction of movement

	.if	0
ExtendLineUp label near
	  mov	dx,offset BackupActiveHistory ;direction of movement
	.endif

	mov	si,cbCurrent		;si= address of current CmdBuf char
	push	[si]			;terminate CmdBuf (for searching)
	mov	byte ptr [si],0
	push	si

	mov	bx,offset slHistory	;bx= history string list structure
	assume	bx:near ptr STRINGLIST
	push	[bx].Active		;search startpoint (for quitting)
	.repeat				;loop to search history,

		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; Change active history row up/down.  If we hit the top or
		; bottom of the history list, wrap to the other end.
		;
		; Note the list may be empty, in which case the below code
		; will do nothing - that's OK, because the string will not
		; be matched below, and we'll exit after it doesn't.

	  mov	si,[bx].Active		;selection ptr prior to movement
	  call	dx			;select previous/next history row
	  .if	si == [bx].Active	;if no movement (at top or bottom),
	    mov	di,[bx].Head		;assume we'll wrap back to start
	    .if	si == di		;if that's where we began,
	      mov di,[bx].Tail		;wrap to bottom instead
	    .endif
	    mov	[bx].Active,di
	  .endif

		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; Match selected entry against portion of CmdBuf.  Note the
		; search routine searches the whole list from di, so if a
		; match is found, we need to insure it matched the first
		; (the selected) list entry.

	  mov	si,offset CmdBuf	;compare CmdBuf to next row
	  mov	di,[bx].Active
	  call	StringListPoolSearch
	  .if	zero?			;if we found a match,
	    cmp	si,[bx].Active		;may only be against selected entry
	  .endif
	  stc				;carry= assume match was made
	  .break .if zero?		;break= matched selected entry

	  pop	ax			;refresh search startpoint
	  push	ax
	.until	ax == [bx].Active	;loop until we've searched entire list
	pop	ax			;ax= search startpoint

	pop	si			;restore CmdBuf image
	pop	[si]

	.if	carry?			;if a match was made above,
	  sub	si,offset CmdBuf	;si= count of orignal prefix chars
	  push	si			;save for cursor reset below
	  call	LoadActiveHistory	;load selected history row
	  call	GotoStartOfLine		;put cursor at start of line
	  pop	cx			;number of chars to advance
	  jmp	ForwardCxChars		;put cursor to where it started out
	.endif
	mov	[bx].Active,ax		;nothing found, don't change active
	assume	bx:nothing

	ret
ExtendLineDown endp

;-----------------------------------------------------------------------------

EnterCommandNumber proc	near
	mov	ax,slHistory.Head	;ignore if history is empty
	cmp	slHistory.Tail,ax
	je	@F

	call	DeleteAll		;erase entire input
	mov	si,offset LinePrompt	;display prompt/insert into CmdBuf
	.repeat
	  lodsb
	  push	si
	  call	StoreChar
	  pop	si
	.until	si == offset LinePrompt + sizeof LinePrompt

	.while	1			;loop to obtain number,
	  call	GetKeybCharNoEcho
	  .continue .if !zero?		;ignore dual-byte keystroke chars
	  .if	al == ESCAPE		;if aborting command,
	    call DeleteAll		;empty command buffer
	    @@:
	    ret				;and exit
	  .endif
	  .break .if al == CR		;stop if CR
	  call	Get_cbCurrent_cbLast	;get cb pointers
	  .if	al == BS		;if backspace,
	    .if	di != offset CmdBuf + sizeof LinePrompt ;if char(s) to delete,
	      call BackSpace		;erase previous char
	    .endif
	    .continue
	  .endif
	  .if al >= '0' && al <= '9'	;if digit,
	    .if di != offset CmdBuf + sizeof LinePrompt + 5 ;if not at max,
	      call StoreChar		;append to buffer
	    .endif
	  .endif
	.endw

	xor	bx,bx			;init number accumulator
	mov	si,offset CmdBuf + sizeof LinePrompt ;start of entry
	.while	1			;loop to convert to binary
	  lodsb
	  sub	al,'0'			;convert next digit
	  .break .if carry?		;break= hit terminating 0
	  cbw				;zero extend
	  xchg	bx,ax			;add to accumulator
	  mul	Constant10
	  .if	!carry?			;if no overflow,
	    add	bx,ax			;cx= new accumulator
	  .endif
	  sbb	dx,dx			;if overflow, keep accumulator at max
	  or	bx,dx
	.endw

	mov	di,slHistory.Head	;select first history entry
	.repeat				;loop to advance to chosen entry,
	  mov	slHistory.Active,di
	  .break .if bx < 2		;break= advancing done
	  dec	bx			;reduce amount left to advance
	  call	GetActiveHistoryRowLength ;chack active history entry
	.until	di == slHistory.Tail	;loop until no more history entries
	jmp	LoadActiveHistory	;load active history row

EnterCommandNumber endp

;-----------------------------------------------------------------------------

ToggleLFN proc near
	  mov	dl,es:cmode
	  and	dl,CM_NOLFN
	  sub	dl,CM_NOLFN
	  neg	dl
	  and	es:cmode,not CM_NOLFN
	  add	es:cmode,dl
	  ret
ToggleLFN endp

ToggleACMode proc near
	  mov	dl,es:cmode
	  and	dl,CM_CHANGE
	  sub	dl,CM_CHANGE
	  neg	dl
	  and	es:cmode,not CM_CHANGE
	  add	es:cmode,dl
	  ret
ToggleACMode endp

ToggleSysHidden proc near
	  mov	dl,es:cmode
	  and	dl,CM_SYSHIDDEN
	  sub	dl,CM_SYSHIDDEN
	  neg	dl
	  and	es:cmode,not CM_SYSHIDDEN
	  add	es:cmode,dl
	  ret
ToggleSysHidden endp

DisplayCompletion proc near
	call	Get_cbCurrent_cbLast
	call	SaveHistory
	mov	bp,-1
	call	TabStart
	call	DeleteAll		;remove all input from screen
	jmp	ExitKeystrokeDispatcher ;exit (force return of empty line)
DisplayCompletion endp

DisplayMacros proc near
	mov	al,0			;indicate macro display

	skip
DisplayHistory label near
	  mov	al,1			;indicate history display
	endskip

	cbw				;extend & save macro/history flag
	push	ax

	call	DeleteAll		;remove all input from screen

	pop	cx			;restore macro/history flag

	mov	ax,slMacros.Head	;assume macro display
	mov	di,offset NumberBuffer + 7
	mov	byte ptr [di],ch
	jcxz	@F			;if assumption wrong,
	  mov	ax,slHistory.Head	;load history address
	@@:
	xchg	bx,ax			;bx= selected string list address

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Ready to display string list.  si= address of selected string list,
	; di= address of work string in NumberBuffer (initialized to null
	; string), cx= 1 (if numbering, this will be the row count), else 0.

	mov	scInfo.CurrentRow,0ffh	;force cursor to 0,0 after next CRLF
	.while	byte ptr [bx] != 0	;while more strings to display,

	  jcxz	@F			;if numbering (history list),
	    mov	di,offset NumberBuffer + 5 ;format end of "nn>:" string
	    mov	ax,'>:'			;assume selected row
	    .if bx != slHistory.Active	;if assumption wrong,
	      mov ah,' '		;correct it
	    .endif
	    mov  word ptr [di],ax
	    mov  ax,cx
	    .repeat			;format in digits
	      xor  dx,dx
	      div  Constant10
	      dec  di
	      add  dl,'0'
	      mov  byte ptr [di],dl
	    .until ax == 0
	    inc  cx			;bump row number for next pass
	  @@:
		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; Display the next string virtually.  If it won't fit
		; onscreen, issue a "-- More --" message before displaying it
		; physically.
		;
		; Note it might have been easier simply to put the -- More --
		; logic in the lowest-level display routine; however, that
		; would lead to multi-row objects potentially being broken up
		; by a "more" message, and I didn't want to let that happen.

	  mov	ax,word ptr scInfo.CurrentCol ;ax= current cursor location
	  .if	ah != 0ffh		;if not top, check if --more-- needed,
	    push ax			;save current cursor location
	    call DisplayCRLFVirtual	;virtual display CRLF
	    mov  si,di			;virtual display number, if numbering
	    call DisplayFormattedMsgVirtual
	    mov  si,bx			;virtual display next string entry
	    call DisplayFormattedMsgVirtual
	    mov  al,scInfo.CurrentRow	;al= post-virtual-display row number
	    pop  word ptr scInfo.CurrentCol ;restore current cursor location
	    inc  ax			;set to TotalRows if we've hit bottom
	    .if  al == scInfo.TotalRows	;if time for -- More -- prompt,
	      mov  si,offset MoreMsg	;display -- More --
	      call DisplayFormattedMsg
	      call GetKeybCharNoEcho	;wait for keystroke
	      .break .if ax == ESCAPE	;break= ESC entered, abort
	      mov  scInfo.CurrentRow,0ffh ;force cursor to 0,0 after next CRLF
	    .endif
	  .endif

	  call	DisplayCRLF		;get to start of next line
	  mov	si,di			;display number, if numbering
	  call	DisplayFormattedMsg
	  mov	si,bx			;display next formatted string,
	  call	DisplayFormattedMsg
	  mov	bx,si			;advance to next list entry
	.endw

	jmp	ExitKeystrokeDispatcher ;exit (force return of empty line)
DisplayMacros endp

;-----------------------------------------------------------------------------
;
; End Module
;
;-----------------------------------------------------------------------------

	ENDSEG
	end	start
