	page	,132

;-----------------------------------------------------------------------------
;
;  This file is part of doskey.com.
; 
;  Copyright (C) 2001-2011 Paul Houle (http://paulhoule.com)
; 
;  This program is free software; you can redistribute it and/or modify
;  it under the terms of the GNU General Public License as published by
;  the Free Software Foundation; either version 2 of the License, or
;  (at your option) any later version.
; 
;  This program is distributed in the hope that it will be useful,
;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;  GNU General Public License for more details.
; 
;  You should have received a copy of the GNU General Public License
;  along with this program; if not, write to the Free Software
;  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
;
;  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;
; Execution proceeds here from command line.  All code and data within is
; temporary (discarded if we install as a TSR) - the area is overlaid by
; the new TSR's BUFSIZE block (used to store the command history and macros).
;
;-----------------------------------------------------------------------------

	.nolist
	include	dkdefs.inc	;include constant definitions, externdef's
	.list
	option	noljmp		;disallow automatic jmp-lengthening

;-----------------------------------------------------------------------------
;
; Jump directly here (to "Startup" label) from DOS entry.
;
; This is a .com image, so all segment registers are the same on entry.
;
; Note: the init code must not alter any run-time variables (variables that
; overlay the PSP), since that would destroy PSP fields that are needed
; during init (e.g., the incoming command line).

Startup:
	;push	es			;090629 *** START
	;xor	di,di			;si:di= Windows API entry address
	;mov	es,di			;  (0:0 if Win 3.1/9x/me not running)
	;mov	ax,1602h		;201408 changed to new approach
	;int	2fh
	;mov	si,es
	;pop	es			;090629 *** END

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Store current int 2fh contents.  This is needed for chaining (in
	; case we go resident), and for uninstalling.

	push	es
	mov	ax,352fh		;save current int 2fh vector
	int	21h
	mov	word ptr Old2fVector,bx
	mov	word ptr Old2fVector+2,es
	pop	es

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Parse CR-terminated command line, and execute appropriate tasks.
	; Note the first thing we do is convert the terminating CR (or 0, if
	; we find that first) into a zero.  This insures we'll NEVER have a
	; zero in the input line, which could be fatal in places.

	mov	si,PSP.pspCommandTail + 1 ;si= addr of incoming command line
	push	si
	.repeat
	  lodsb				;scan to CR or 00
	  .break .if al == 0
	  xor	al,CR
	.until	zero?
	mov	[si-1],al		;terminate/re-terminate with a zero
	pop	si

	.repeat				;start dummy block (to support breaks)

	  ifdef	EBUG
		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; If debug build, issue a warning message.
		;
		; Also, check that the assembly went OK.  This should have
		; been done at assemble time, but MASM 6.11 screws up when
		; evaluating differences between labels on the first "pass."
		; So we do the check at run time during debug builds only.
		;
		; Lastly, shrink our memory block paras to 1 max segment's
		; worth.  This eases debugging a bit by allowing a command
		; shell to be entered during a debug session.

	    push si
	    mov	 si,offset WarDebMsg
	    call DisplayFormattedMsg
	    pop  si
	    mov	cx,offset InstallStackBase ;cx + install stack (256 bytes)
	    sub	cx,offset EndResident	   ;  must fit
	    mov	di,offset GoTSRTooBigMsg ;assume GoTSR overlay region overflow
	    mov	ax,GF_EXIT or GF_ERROR
	    .break .if cx > (BUFSIZE_MIN - MACROS_MAX) ;break= overflow, abort
	    mov	bx,sp
	    add bx,15
	    rcr bx,1
	    mov cl,3
	    shr bx,cl
	    mov ah,4ah
	    int 21h
	  endif

	  mov cmode,0
	  call	ParseOpts		;parse any /options
	  .break .if al & GF_EXIT	;break= exit immediately w/message
	  call	AdjustBufsize		;condition final BUFSIZE value
	  xor	bp,bp
	  call	ParseMacroDef		;parse macro definiton
	  .break .if al & GF_EXIT	;break= exit immediately w/message
	  .if	!(al & GF_REINSTALL)	;if not forcing reinstall,
	    call ModifyTSR		;attempt to modify an installed TSR
	  .endif
	  .if cmode & CM_LOWER && cmode & CM_UPPER
	    and cmode, not CM_LOWER
	    and cmode, not CM_UPPER
	  .endif
	  .if	ax & GF_VERBOSE		;if status display requested,
	    push si
	    push es
	    push ax
	    call ShowStatus
	    pop ax
	    pop es
	    pop si
	  .endif
	  .if	al & GF_REINSTALL	;if reinstalling or nothing to modify,
	    and	al,TF_RESIDENTFLAGS	;init the resident flags
	    mov	tflags,al
	    mov	si,offset DoskeyInstalledMsg ;issue installation message now
	    call DisplayFormattedMsg
	    call GetMacroDefinitionInfo	;get macro info for InstallTSR
	    jmp	InstallTSR		;jmp= install-new/reinstall TSR
	  .endif
	.until	1			;end dummy block

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Exit points.  ax= flags, di= potential exit message address.

	.if	al & GF_EXIT		;if exit forced w/message,
	  push	ax			;save flags
	  mov	si,di			;display exit message
	  call	DisplayFormattedMsg
	  pop	ax			;restore flags
	.endif
	and	al,GF_ERROR		;set non-zero errolevel, if needed
	mov	ah,4ch			;exit to OS
	int	21h
	
ChVars:
	mov bl,cmode
	push bx
	.if	ax & GF_LFN || ax & GF_NOLFN
	  and	bl,CM_NOLFN
	  and	es:cmode,not CM_NOLFN
	  add	es:cmode,bl
	.endif
	pop bx
	push bx
	.if	ax & GF_APPEND || ax & GF_CHANGE
	  and	bl,CM_CHANGE
	  and	es:cmode,not CM_CHANGE
	  add	es:cmode,bl
	.endif
	pop bx
	push bx
	.if	ax & GF_NOSYSHIDDEN || ax & GF_SYSHIDDEN
	  and	bl,CM_SYSHIDDEN
	  and	es:cmode,not CM_SYSHIDDEN
	  add	es:cmode,bl
	.endif
	pop bx
	push bx
	.if es:cmode & CM_INSERT && !(bl & CM_STARTCHANGE) && bl & CM_UPPER
	  or	ax,TF_STARTINSERT_CHANGE or TF_STARTINSERT
	  or	bl,CM_STARTCHANGE
	  or	bl,CM_INSERT
	.endif
	.if	bl & CM_STARTCHANGE && ax & TF_STARTINSERT_CHANGE
	  and	bl,CM_INSERT
	  and	es:cmode,not CM_INSERT
	  add	es:cmode,bl
	.endif
	pop bx
	push bx
	.if	bl & CM_LOWER && ax & GF_LOWER && !(bl & CM_UPPER)
	  and	bl,CM_LOWER
	  and	es:cmode,not CM_LOWER
	  and	es:cmode,not CM_UPPER
	  add	es:cmode,bl
	.endif
	pop bx
	push bx
	.if	bl & CM_UPPER && ax & GF_UPPER && !(bl & CM_LOWER)
	  and	bl,CM_UPPER
	  and	es:cmode,not CM_LOWER
	  and	es:cmode,not CM_UPPER
	  add	es:cmode,bl
	.endif
	pop bx
	.if	bl & CM_LOWER && bl & CM_UPPER && ax & GF_LOWER && ax & GF_UPPER
	  and	bl, not CM_LOWER
	  and	bl, not CM_UPPER
	  and	es:cmode,not CM_LOWER
	  and	es:cmode,not CM_UPPER
	.endif
ret

;-----------------------------------------------------------------------------
; Attempt to modify an installed TSR (caller did not specify /REINSTALL).
; Note: if no version of DOSKEY (compatible or otherwise), is currently
; installed, we force /REINSTALL and exit immediately.
;
; IN:
;   ax= parsed option flags (GF_REINSTALL bit must be clear)
;   SwiData1.iisSize= parsed BUFSIZE
; OUT:
;   ax= parsed option flags: GF_REINSTALL turned on if no TSR installed;
;	otherwise, GF_EXIT/GF_ERROR may be set if an error occurs.
;   di= exit message, if ax & GF_EXIT

ModifyTSR proc	near
	push	es			;save local es
	push	ax			;save parsed option flags

	.repeat				;start dummy block (to support breaks)
	  mov	ax,4800h		;perform DOSKEY installation check
	  int	2fh
	  mov	bl,GF_REINSTALL		;assume we'll force a reinstall
	  .break .if ax == 4800h	;break= not installed, force reinstall
	  mov	bl,GF_EXIT or GF_ERROR	;now assume incompatible TSR error
	  mov	di,offset IncompatDoskeyMsg
	  .break .if ax != DOSKEY_ID	;break= incompatible TSR

		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; Compatible TSR already installed (es= installed TSR
		; segment).  Attempt to modify the installed TSR segment.
		;
		; Only allow modification if user hasn't attempted to
		; specify a different BUFSIZE.

	  pop	ax			;restore flags image
	  push	ax
	  .if	ax & GF_BUFSIZE		;if user specified /BUFSIZE,
	    mov	di,offset es:SwiData1.iisSize
	    mov	bl,GF_EXIT or GF_ERROR
	    mov	si,offset SwiData1.iisSize	;test assumption
	    mov	di,si
	    cmpsw
	    .if !zero?
	      mov di,offset CantChangeBufsizeMsg	;BUFSIZE change error
	      .break		;break= assumption correct, abort
	    .endif
	  .endif

		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; Change any resident flags requested by user.

	  call	ChVars
	  and	ax,TF_RESIDENTFLAGS or (TF_RESIDENTFLAGS shl 8)
	  not	ah
	  and	ah,es:tflags
	  or	al,ah
	  mov	es:tflags,al

		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; Process macro definition/redefinition/deletion.

	  call ProcessMacro
	  .break .if cx && bl != 0	;break= macro storage error
	  pop	ax			;restore flags image
	  push	ax

	  ifdef	EBUG
		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; If requested, simulate an int 2fh get input line call,
		; using this code (so we can debug), but using the data
		; segment of the installed TSR.

	    .if	ax & GF_GETINPUT
		push ax			;save flags
		call GetInput		;set up and simulate int 2f get input
		pop  ax			;restore flags
	    .endif
	  endif

		;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; Display installed history and/or macros, if requested.

	  push	ds
	  push	es
	  pop	ds
	  .if	al & GF_HISTORY		;if history display requested,
	    push ax			;save option flags
	    mov  si,slHistory.Head	;display installed command history
	    call ListDisp
	    pop  ax			;restore option flags
	  .endif
	  .if	al & GF_MACROS		;if macros display requested,
	    mov  si,slMacros.Head
	    call ListDisp
	  .endif
	  pop	ds

	  xor	bx,bx			;indicate TSR work successful
	.until	1			;end dummy block

	pop	ax			;restore parsed option flags
	pop	es			;restore local es
	or	al,bl			;set any flags selected above

	ret
ModifyTSR endp


	.repeat
	  call DisplayFormattedMsg	;display next formatted string
	  call OutputCRLF		;follow with a CRLF
ListDisp label near
	.until byte ptr [si] == 0
	retn

ifdef	EBUG
;-----------------------------------------------------------------------------
; Simulate an int 2fh get input line call, using this code (so we can debug),
; but using the data segment of the installed TSR.

GetInput proc near
	push	es			;move in test command line template
	push	ds
	pop	es
	mov	di,offset CmdBufLen
	mov	si,offset TestCmdTemplate
	mov	cx,(TestCmdTemplateEnd - TestCmdTemplate) + 1
	rep movsb
	pop	es

	mov	dx,offset CmdBufLen	;ds:dx= incoming template
	mov	bx,'de'			;set special DEBUG signature
	mov	cx,'ug'
	mov	ax,4810h		;int 2fh DOSKEY get command line
	pushf				;create iret frame
	push	cs
	call	near ptr MyInt2f	;go perform simulation

	mov	si,offset CmdBufLen + 1	;si= returned input line length
	lodsb				;ax= command length, w/o CR
	mov	ah,0
	add	si,ax			;point to CR
	mov	word ptr [si],']'	;terminate with ']',0
	mov	si,offset CmdBufLen + 2	;display entered command line
	push	es
	push	ds
	pop	es
	call	DisplayFormattedMsg
	pop	es

	ret
GetInput endp
endif

ShowStatus:
	xor	bp,bp
	.if al & GF_REINSTALL
	  inc bp
	.endif
	mov	ax,4800h
	int	2fh				;get the DOSKEY instance
	.if ax != DOSKEY_ID && !bp
	  ret
	.endif
	mov	si,offset CurrentStatusMsg
	call	DisplayFormattedMsg
	mov	dl,es:cmode
	mov	dh,cmode
	mov	si,offset LFNSupportMsg
	call	DisplayFormattedMsg
	mov	si,offset ONMsg
	.if dh & CM_NOLFN && bp || dl & CM_NOLFN && !bp
	  mov	si,offset OFFMsg
	.endif
	call	DisplayFormattedMsg
	mov	si,offset AppendModeMsg
	.if dh & CM_CHANGE && bp  || dl & CM_CHANGE && !bp
	  mov	si,offset ChangeModeMsg
	.endif
	call	DisplayFormattedMsg
	mov	si,offset ONMsg
	call	DisplayFormattedMsg

	mov si,offset ToLowerMsg
	call	DisplayFormattedMsg
	mov	si,offset OFFMsg
	.if dh & CM_LOWER && bp || dl & CM_LOWER && !bp
	 mov	si,offset ONMsg
	.endif
	call	DisplayFormattedMsg
	mov si,offset ToUpperMsg
	call	DisplayFormattedMsg
	mov	si,offset OFFMsg
	.if dh & CM_UPPER && bp || dl & CM_UPPER && !bp
	 mov	si,offset ONMsg
	.endif
	call	DisplayFormattedMsg

	mov	si,offset ShowSysHiddenMsg
	call	DisplayFormattedMsg
	mov	si,offset OFFMsg
	.if dh & CM_SYSHIDDEN && bp || dl & CM_SYSHIDDEN && !bp
	  mov	si,offset ONMsg
	.endif
	call	DisplayFormattedMsg
	mov	si,offset OverwriteModeMsg
	mov	dl,es:tflags
	pop	ax
	push	ax
	.if dh & CM_INSERT && bp || dl & TF_STARTINSERT && !bp
	  mov	si,offset InsertModeMsg
	.endif
	call	DisplayFormattedMsg
	mov	si,offset ONMsg
	call	DisplayFormattedMsg
	mov	si,offset BufSizeMsg
	call	DisplayFormattedMsg
	.if bp
	  mov	bp,SwiData1.iisSize
	.else
	  mov	bp,es:SwiData1.iisSize
	.endif
	xor	dx,dx
	.while bp >= 3e8h
	  sub bp,3e8h
	  inc dx
	.endw
	.if dx > 0
	  push bp
	  mov bp,dx
	  xor ax,ax
	  .if bp >= 0ah
	    mov dh,0ah
	    call DispNumber
	  .endif
	  mov dh,1
	  call DispNumber
	  pop bp
	.endif
	xor ax,ax
	mov dh,64h
	call DispNumber
	mov dh,0ah
	call DispNumber
	mov dh,1
	call DispNumber
	call DisplayCRLF
	ret

DispNumber:
	sub bp,ax
	mov ax,bp
	div dh
	push ax
	add al,30h
	call DisplayFormattedAl
	pop ax
	mul dh
	ret
;-----------------------------------------------------------------------------
; Parse command line options, if any.
;
; IN:
;   si= start of 0-terminated command line.
; OUT:
;   ax= parsed option flags
;   di= exit message, if ax & GF_EXIT

ParseOpts proc	near
	xor	ax,ax			;various option flags accumulated here
	push	ax
	.repeat				;loop to parse all options,

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Scan option table for match of option string.
	  call	SkipWhitespace		;skip any leading whitespace
	  .break .if al != '-' && al != '/' ;break= no more options found
	  inc	si			;bypass option indicator

	  push	si			;save option start
	  .repeat			;scan past all option name text
	    lodsb
	    .if	al != '?'		;allow '?' in option name
	      call IsMacroNameChar	;also allow macro name characters
	    .endif
	  .until !zero?
	  dec	si			;point to terminator
	  pop	di			;restore option start
	  push	[si]			;save terminating text
	  push	si			;save address of terminator
	  mov	byte ptr [si],0		;make option asciiz string

	  mov	si,offset OptionTableList ;start of option table
	  .while 1			;loop to scan option table,
	    lodsw			;bx= option handler address
	    xchg bx,ax
	    push si			;save current option table address
	    push di			;save option text address
	    .if  byte ptr [si]		;if haven't hit end of option table,
	      call StringListPoolSearch	;check for option match
	    .endif
	    pop  di			;restore option text address
	    pop	 si			;restore current option table address
	    push ax
	    pushf
	    mov al,[di]
	    call ToLower
	    mov dl,al
	    popf
	    pop ax
	    .break .if zero? && dl==[si] || !byte ptr [si] ;break= match, or end of table
	    .repeat			;scan to start of next option
	      lodsb
	    .until al == 0
	  .endw

	  pop	si			;si= terminator address
	  pop	word ptr [si]		;restore terminator characters

	  pop	ax			;put flags in ax for option handler
	  call	bx			;execute option handler
	  test	al,GF_EXIT		;non-zero if exit was forced
	  push	ax
	.until	!zero?			;loop if exit was not forced
	pop	ax			;ax= parsed option flags
	ret
ParseOpts endp

;-----------------------------------------------------------------------------

GiveHelp proc	near
	mov	di,offset HelpMsg	;address of message to display
	mov	al,GF_EXIT		;force immediate exit
	ret
GiveHelp endp

;-----------------------------------------------------------------------------

SetInsert proc near
	or	ax,TF_STARTINSERT_CHANGE or TF_STARTINSERT
	or	cmode,CM_STARTCHANGE
	or	cmode,CM_INSERT
	ret
SetInsert endp

SetOverstrike proc near
	and	al,not TF_STARTINSERT
	or	ax,TF_STARTINSERT_CHANGE
	or	cmode,CM_STARTCHANGE
	and	cmode,not CM_INSERT
	ret
SetOverstrike endp

;-----------------------------------------------------------------------------

SetReinstall proc near
	or	al,GF_REINSTALL
	ret
SetReinstall endp

;-----------------------------------------------------------------------------

SetMacros proc near
	or	al,GF_MACROS
	ret
SetMacros endp

;-----------------------------------------------------------------------------

SetHistory proc near
	or	al,GF_HISTORY
	ret
SetHistory endp

DisableLFN proc near
	or	ax,GF_NOLFN
	or	cmode,CM_NOLFN
	ret
DisableLFN endp

EnableLFN proc near
	or	ax,GF_LFN
	and	cmode,not CM_NOLFN
	ret
EnableLFN endp

AppendMode proc near
	or	ax,GF_APPEND
	and	cmode,not CM_CHANGE
	ret
AppendMode endp

ChangeMode proc near
	or	ax,GF_CHANGE
	or	cmode,CM_CHANGE
	ret
ChangeMode endp

NoSysHidden proc near
	or	ax,GF_NOSYSHIDDEN
	and	cmode,not CM_SYSHIDDEN
	ret
NoSysHidden endp

SysHidden proc near
	or	ax,GF_SYSHIDDEN
	or	cmode,CM_SYSHIDDEN
	ret
SysHidden endp

ToLowerCase proc near
	or	ax,GF_LOWER
	or	cmode,CM_LOWER
	ret
ToLowerCase endp

ToUpperCase proc near
	or	ax,GF_UPPER
	or	cmode,CM_UPPER
	ret
ToUpperCase endp

Verbose proc near
	or	ax,GF_VERBOSE
	ret
Verbose endp

;-----------------------------------------------------------------------------
;
; Uninstall logic.  If we recognize an installed DOSKEY, and the int 2f
; vector still points to it, we allow the uninstall; otherwise, immediate
; exit w/error message.

Uninstall proc	near

	push	es
	.repeat			;start dummy block (to support breaks)
	  mov	ax,4800h		;check if DOSKEY installed; if so,
	  int	2fh			;  es= installed segment address
	  cmp	ax,DOSKEY_ID		;zero status if signature correct
	  mov	di,offset CantUninstallMsg	;assume "can't uninstall"
	  mov	ax,GF_EXIT or GF_ERROR		;  message w/errorlevel 1
	  mov	cx,es			;break= int 2fh vector has changed
	  .break .if word ptr Old2fVector + 2 != cx
	  .break .if word ptr Old2fVector != offset MyInt2f

	  push	ds			;restore installed DOSKEY old int 2fh
	  lds	dx,es:Old2fVector	;  vector
	  mov	ax,252fh
	  int	21h
	  pop	ds

	  mov	ah,49h			;release installed memory block
	  int	21h

	  mov	di,offset DoskeyUninstalledMsg	;issue uninstall message
	  mov	ax,GF_EXIT			;  and exit w/0 errorlevel
	.until	1		;end dummy block
	pop	es
	ret

Uninstall endp

;-----------------------------------------------------------------------------

BufSize proc near
	push	ax		;save flags

	call	SkipWhitespace	;skip any whitespace following /BUFSIZE
	stc			;(set carry in case it isn't)
	.if al == '=' || al == ':' ;if option formed OK so-far,
	  inc	si		;bypass '=' or ':'
	  call	SkipWhitespace
	  xor	cx,cx		;accumulate number here
	  .repeat		;loop to parse numbers
	    lodsb		;ax= next character, in numeric form
	    cbw			;(zero ah)
	    sub	al,'9' + 1	;set carry if next character is a digit
	    add	al,10
	    .break .if !carry?	;break= no more digits, done w/number
	    xchg ax,cx		;multiply accumulator by 10
	    mov	dx,10
	    mul	dx
	    .break .if carry?	;break= 16-bit overflow
	    add	cx,ax		;cx= updated accumulator
	  .until carry?		;continue if no 16-bit overflow
	  dec	si		;back up in front of terminating character
	  mov	SwiData1.iisSize,cx ;save parsed buffer size, good or bad
	.endif

	pop	ax		;restore flags
	.if	carry?		;if error parsing BUFSIZE,
	  mov	di,offset BadBufSizeSettingMsg	;cause exit with message
	  mov	al,GF_EXIT or GF_ERROR		;  and non-zero errorlevel
	.endif
	or	ax,GF_BUFSIZE	;show BUFSIZE was parsed

	ret
BufSize endp

;-----------------------------------------------------------------------------

ifdef	EBUG
SetGetInput proc near
	or	ax,GF_GETINPUT
	ret
SetGetInput endp
endif

;-----------------------------------------------------------------------------

InvalidOption proc	near
	mov	word ptr [si],CR		;terminate option w/CR,0

	mov	si,offset InvalidOptionMsg	;issue invalid option message
	call	DisplayFormattedMsg

	dec	di				;cause exit with message
	mov	al,GF_EXIT or GF_ERROR		;  and non-zero errorlevel
	ret
InvalidOption endp

;-----------------------------------------------------------------------------
; Adjust BUFSIZE.  We limit the minimum to one worst-case history and
; macro row, and the maximum to the amount of space in our installed
; segment.  Also, we round the final value up to use a multiple of
; paragraphs, so we don't waste a few bytes of space at the end of the
; segment
;
; IN:
;   ax= parsed option flags
;   si= current command line position
;   SwiData1.iisSize= entry BUFSIZE
; OUT:
;   ax= parsed option flags (no change)
;   si= current command line position (no change)
;   SwiData1.iisSize= adjusted BUFSIZE

AdjustBufsize proc near

	push	ax			;save parsed option flags

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Limit /BUFSIZE to a minimum of BUFSIZE_MIN,
	; and a maximum of the amount of memory in the entry .com segment.

	mov	ax,SwiData1.iisSize
	.if	ax < BUFSIZE_MIN	;limit BUFSIZE minimum
	  mov	ax,BUFSIZE_MIN
	.endif
	mov	dx,sp			;limit BUFSIZE maximum
	sub	dx,offset EndResident
	.if	ax > dx
	  xchg	ax,dx
	.endif

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Whatever the BUFSIZE is now, round it up so we use all of the
	; final paragraph in the segment.

	add	ax,offset EndResident + 0fh ;round to next paragraph boundary
	sbb	dx,dx			;limit to 1 pargraph under 64k
	or	ax,dx
	and	al,not 0fh
	sub	ax,offset EndResident
	mov	SwiData1.iisSize,ax

	pop	ax			;restore parsed option flags

	ret
AdjustBufsize endp

;-----------------------------------------------------------------------------
; Table used to parse command-line options.

DECLARE_OPTION	macro	string,handler
	dw	offset handler
	db	string,0
	endm

OptionTableList label	near	;marks start of option string list
	DECLARE_OPTION	'?',GiveHelp
	DECLARE_OPTION	'i',SetInsert
	DECLARE_OPTION	'o',SetOverstrike
	DECLARE_OPTION	'r',SetReinstall
	DECLARE_OPTION	'h',SetHistory
	DECLARE_OPTION	'm',SetMacros
	DECLARE_OPTION	'u',Uninstall
	DECLARE_OPTION	'b',BufSize
	DECLARE_OPTION	'd',DisableLFN
	DECLARE_OPTION	'e',EnableLFN
	DECLARE_OPTION	'a',AppendMode
	DECLARE_OPTION	'c',ChangeMode
	DECLARE_OPTION	'n',NoSysHidden
	DECLARE_OPTION	's',SysHidden
	DECLARE_OPTION	'l',ToLowerCase
	DECLARE_OPTION	'p',ToUpperCase
	DECLARE_OPTION	'v',Verbose
	ifdef	EBUG
	  DECLARE_OPTION 'g',SetGetInput
	endif
	dw	InvalidOption		;terminate option string list
	db	0

;-----------------------------------------------------------------------------
;
; Message strings.  Each message must be terminated with a zero.

ifdef	EBUG
  TestCmdTemplate db IOBufSize
	db	TestCmdTemplateEnd - TestCmdTemplateStart
  TestCmdTemplateStart db 'Test input line'
  TestCmdTemplateEnd   db 0dh
  GoTSRTooBigMsg db '*** Assemble-time error: overlay region too big ***',CR,0
  WarDebMsg   db '*** Warning: debug build ***',CR,0
endif

CantChangeBufsizeMsg db 'Cannot change BUFSIZE of installed DOSKEY.',CR,0

DoskeyInstalledMsg db 'DOSKEY installed.',CR,0

InvalidMacroDefMsg db 'Invalid macro definition.',CR,0

InvalidOptionMsg db 'Invalid option: ',0

BadBufSizeSettingMsg db 'Invalid /BUFSIZE=size specification.',CR,0

CantUninstallMsg label byte
 db 'Cannot Uninstall: DOSKEY not resident, vectors have changed, or',CR
IncompatDoskeyMsg label byte	;(share message text)
 db 'An incompatible DOSKEY is installed.',CR,0

DoskeyUninstalledMsg db 'DOSKEY uninstalled.',CR,0

CurrentStatusMsg db 'DOSKEY current settings:',CR,0
LFNSupportMsg db 'Long file name support ',0
AppendModeMsg db 'Auto-completion append mode ',0
ChangeModeMsg db 'Auto-completion change mode ',0
ShowSysHiddenMsg db 'Including system/hidden files ',0
ToLowerMsg db 'Auto-conversion to lower case ',0
ToUpperMsg db 'Auto-conversion to upper case ',0
InsertModeMsg db 'Keystroke insert mode ',0
OverwriteModeMsg db 'Keystroke overwrite mode ',0
BufSizeMsg db 'Command recall/macro size: ',0
ONMsg	db 'ON.',CR,0
OFFMsg	db 'OFF.',CR,0

HelpMsg label near
			;20030121 Version changed to 1.6, copyright to 2003
			;20030926 Version changed to 1.7, added paulhoule.com
			;20031022 Version changed to 1.8
			;20090612 Version changed to 1.9, copyright to 2009
			;20090629 Version changed to 2.0
			;       Version changed to 2.1, LFN support, 2014
			;20140725 Version changed to 2.2
			;       Version changed to 2.5, new options and features
			;20170516 Version changed to 2.6
			;20170627 Version changed to 2.7
 db CR
 db 'Enhanced DOSKEY Ver 2.7b - edits, recalls, auto-completes commands.',CR
 db 'Copyright 2009 Paul Houle (paulhoule.com).  All rights reserved.',CR
 db 'Copyright 2017 Wengier for LFN support and other enhancements.',CR,CR
 db 'DOSKEY [-option [-option]] [macro=[text]]',CR,CR
 db '  -?               This help message',CR
 db '  -R{einstall}     Install new instance of DOSKEY',CR
 db '  -U{ninstall}     Remove DOSKEY from memory',CR
 db '  -B{ufsize}=size  Set command recall/macro size (default: ~800)',CR
 db '  -D{isableLFN}    Disable long file name support',CR
 db '  -E{nableLFN}     Enable long file name support (default)',CR
 db '  -A{ppendMode}    Append to completed name (default)',CR
 db '  -C{hangeMode}    Change to match full name automatically',CR
 db '  -N{oSystemFiles} Exclude system and hidden files (default)',CR
 db '  -S{ystemFiles}   Include system and hidden files',CR
 db '  -L{owercase}     Auto-conversion to lower case',CR
 db '  -P{upercase}     Auto-conversion to upper case',CR
 db '  -M{acros}        Display macros',CR
 db '  -H{istory}       Display command recall buffer',CR
 db '  -I{nsert}        Keystroke insert mode',CR
 db '  -O{verstrike}    Keystroke overwrite mode (default)',CR
 db '  -V{erbose}       Display current settings',CR
ifdef	EBUG
 db '  -G{etinput}  *** DEBUG: Simulate int 2fh get input line call',CR
 db '               ***        Uses already-installed TSR segment',CR
endif
 db '  macro=           Name of macro to modify or delete',CR
 db '  [text]           Commands to record (empty to delete macro)',CR
 db '  The format "macro=[text]" (without quotation marks) can be',CR
 db '  directly used in command line too when DOSKEY is installed.',CR,CR
 db 'ESC clears command; UP/DOWN selects history; LEFT/RIGHT navigates within the',CR
 db 'command line; ALT+F5 toggles long file name support; ALT+F6 generates a tab;',CR
 db 'F7 displays history; ALT+F7 clears history; F8/ALT+F8 selects history; F9',CR
 db 'selects command by number; ALT+F9 toggles between append mode and change mode;',CR
 db 'F10 displays macros; ALT+F10 clears macros; F11 shows all matching commands',CR
 db 'or file names; ALT+F11 toggles the inclusion of system and hidden files;',CR
 db 'TAB/CTRL+TAB/SHIFT+TAB auto-completes partial command or file name.',CR
 db 'Function keys F1-F12 can be redefined using f1[!]-f12[!] macros.',CR,CR
 db 'The following codes are allowed in macros:',CR
 db '$T           Separates commands within a macro.',CR
 db '$L,$G,$B,$$  Replaced with <, >, |, $ when macro executed.',CR
 db '$0-$9        Parameter replacement (similar to batch file %0-%9).',CR
 db '$*           Replaced with entire line following macro name.',CR,0

;-----------------------------------------------------------------------------
;
; End Module
;
;-----------------------------------------------------------------------------

	ENDSEG
	end
