/*
	phcomp.c - Compares two file w/progress timer.  PaulHoule.com

	compiled MSVC 7.1:	cl /Oxs phcomp.c kernel32.lib user32.lib
						(include=...;msvc\sdk\include  lib=...;msvc\sdk\lib)
	then UPX'd 3.91w:	upx phcomp.exe

	Change Log:
		11/21/2014 Ver 1.5:	Added -o option (set starting file offset).
							Increased default bufsz x 8 (better speed).

		11/22/2014 Ver 1.6:	Allowed -o to be non-ALGN value (start anywhere).
							Added -m option (max bytes read).
							Increased default bufsz to 4096 x 4096 (16 meg).
							Replaced some C stdlib calls with equiv. WIN32.

		 1/ 6/2015 Ver 1.7:	Added -d option.
		 					Allowed 2nd filespec to be a path.
							Allowed option numbers to contain ignored commas.
							Fixed -o bug if offset was > filesize.
							Replaced some C stdlib calls with equiv. WIN32.

	-m w/no arg could mean smaller file size...
*/

#include <stdio.h>				// (fputs, stdout, stderr, fileno)
#include <io.h>					// (isatty)
#include <windows.h>

#define ALGN 4096				// buffer alignment, and minimum buffer size
#define ALGN_DN(x) ((x) & ~(ALGN-1))
#define ALGN_UP(x) ALGN_DN((x) + (ALGN-1))
#define U64(i) ((unsigned __int64)(i))
#define MYEXIT(xc) exit(xc)

int bufsz = 4096 * ALGN;		// default I/O buffer size
__int64 pos = 0;				// current position in both files
int dispf = 0;					// non-zero to disp filename in progress msg
__int64 mfsiz = (U64(1)<<63)-1;	// file size limit (used for -m only)

char wspbuf[1040];				// wsprintf() buffer

void *gmem(unsigned bytes) {
	void *pmem = HeapAlloc(GetProcessHeap(), 0, bytes);
	if (!pmem) MYEXIT(20);
	return pmem;
}

void derror(char *fn, char *ftext, int ecode, int xcode) {
	wsprintf(wspbuf, "*** Error %d %s '%s'\n", ecode, ftext, fn);
	fputs(wspbuf, stdout);
	MYEXIT(xcode);
}

HANDLE dopenr(char *fn) {
	return CreateFile(fn,			// file name
		GENERIC_READ,				// open for reading
		FILE_SHARE_READ,			// share for reading
		NULL,						// default security
		OPEN_EXISTING,				// existing file only
		FILE_FLAG_NO_BUFFERING,		// don't use/alter Windows disk cache
		NULL						// no attr. template
	);
}

void dread(char *fn, HANDLE handle, char *buf, int size, int *rsize) {
	int res = ReadFile(handle, buf, ALGN_UP(size), rsize, NULL);
	if (*rsize > size) *rsize = size;
	if (res) return;
	derror(fn, "reading", GetLastError(), 11);
}

char *nfmt(char *pEnd, unsigned __int64 n) {
	char *pWrk = pEnd;
	int c = 3;
	*--pWrk = (char) 0;
	do {
		if (--c < 0) { *--pWrk = ','; c += 3; }
		*--pWrk = (char) ('0' + n%10);
	} while (n /= 10);
	return pWrk;
}

int memcmp__i32(char *p1, char *p2, register int cnt) {
										// fix alignment if needed
	while (p1 - (char *)0 & 0xf && cnt && *p1 == *p2) p1++, p2++, cnt--;
	while ((cnt -= 32) >= 0) {			// optimize if enough data
		#define MCD(i) if (((__int32 *)p1)[i] != ((__int32 *)p2)[i]) break;
		MCD(0) MCD(1) MCD(2) MCD(3) MCD(4) MCD(5) MCD(6) MCD(7)
		p1 += 32, p2 += 32;
	}
	cnt += 32;
	if (cnt) do {						// do last handful of bytes slowly
	} while (*p1++ == *p2++ && --cnt);
	return cnt;
}

__int64 toi(char *popt) {
	__int64 r = 0;
	char nx;
	while (nx = *popt++) {
		if (nx == ',') continue;
		if (nx < '0' || nx > '9') return -1;
		if (r > ((U64(1)<<63)-1)/10 ) return -1;
		r = r * 10 + (nx - '0');
	}
	return r;
}

void do_one_opt(char *popt) {
	char ol = (char) CharLower((LPTSTR) (popt[1] & 0xff));
	if (ol == 'd') {
		__int64 tbf;
		dispf = 255;
		if (!popt[2]) return;
		dispf = (int) (tbf = toi(&popt[2]));
		if (tbf >= 0 && tbf <= 255) return;
	}
	if (ol == 'b') {
		__int64 tbf = toi(&popt[2]);
		if (tbf >= 0 && tbf <= (U64(1)<<31)-1) {
			bufsz = (int) tbf;
			return;
		}
	}
	if (ol == 'o') {
		if ((pos = toi(&popt[2])) >= 0) return;
	}
	if (ol == 'm') {
		if ((mfsiz = toi(&popt[2])) >= 0) return;
	}
	if (ol == '?') {
		char nbuf[40];
		wsprintf(wspbuf,
			"Compares files w/timer.  Ver 1.7, 1/6/2015 PaulHoule.com\n"
			"If second file omitted, first file is read and discarded.\n\n"
			"Bypasses operating system cache to produce consistent timings\n"
			"(and a true test of the underlying storage medium).\n"
			"\n Usage:  phcomp {-opt(s)} file1 {file2|path}\n\n"
			"-?     This display.\n"
			"-Dnnn  Display (nnn chars of) file name in progress message.\n"
			"-Bnnn  Set read buffer size (default is %s).\n"
			"-Onnn  Set start file offset(s) to nnn.\n"
			"-Mnnn  Set max bytes read for file(s) to nnn.\n"
			, nfmt(&nbuf[sizeof(nbuf)], bufsz)
		);
		fputs(wspbuf, stdout);
		MYEXIT(1);
	}
	wsprintf(wspbuf, "bad opt: %s\n", popt);
	fputs(wspbuf, stderr);
	MYEXIT(99);
}

void do_opts(int *pargc, char *argv[]) {
	int j = 0, i = 0;
	while (++i < *pargc) {
		if (argv[i][0] == '-') do_one_opt(argv[i]);
		else argv[++j] = argv[i];
	}
	*pargc = ++j;
}

char *ffn(char *pfn) {					// isolate file name (skip over path)
	char *pfe = pfn;
	while (*pfe++) ;
	while (--pfe > pfn && pfe[-1] != '\\' && pfe[-1] != ':') ;
	return pfe;
}

main(int argc, char *argv[]) {
	HANDLE handle1;						// handle for 1st file
	HANDLE handle2;						// handle for 2nd file
	char *pfnx, *pfn2;					// various ptrs to file names
	char *pbf1, *pbf2;					// ptr to buffers
	DWORD OpenTick;						// GetTickCount() after file open(s)
	int chunk1;							// size of last chunk read from file1
	int chunk2;							// size of last chunk read from file2
	int chunkx;							// smallest chunk for matching
	int percsec;
	int sskip;							// compare skip (for -o only)
	__int64 maxb;

	do_opts(&argc, argv);				// parse options, if any
	if (argc < 2) do_one_opt("-?");		// give help if no filenames

	pfnx = ffn(argv[1]);				// isolate file name (beyond any path)
	handle1 = dopenr(argv[1]);			// open 1st file
	if (handle1 == INVALID_HANDLE_VALUE)
		derror(argv[1], "opening", GetLastError(), 10);
	*(DWORD *)&maxb = GetFileSize(handle1, 1 + (DWORD *) &maxb);
	pbf1 = (char *) ALGN_UP((char *)gmem(bufsz + ALGN) - (char *)0);
	if (argc > 2) {						// open 2nd file, if there
		__int64 f2siz;
		DWORD fa = GetFileAttributes(pfn2 = argv[2]);
		if (fa != INVALID_FILE_ATTRIBUTES && fa & FILE_ATTRIBUTE_DIRECTORY) {
			int dl;
			pfn2 = gmem(2048);
			CopyMemory(pfn2, argv[2], dl = lstrlen(argv[2]));
			if (pfn2[dl-1] != ':' && pfn2[dl-1] != '\\') pfn2[dl++] = '\\';
			lstrcpy(pfn2 + dl, pfnx);
		}
		handle2 = dopenr(pfn2);
		if (handle2 == INVALID_HANDLE_VALUE)
			derror(pfn2, "opening", GetLastError(), 10);
		*(DWORD *)&f2siz = GetFileSize(handle2, 1 + (DWORD *) &f2siz);
		pbf2 = (char *) ALGN_UP((char *)gmem(bufsz + ALGN) - (char *)0);
		if (maxb > f2siz) maxb = f2siz;
	}
	if (maxb > mfsiz) maxb = mfsiz;		// limit size if option specified

	if (pos > maxb) pos = maxb;
	sskip = (int)pos & (ALGN-1);
	pos &= ~(ALGN-1);
	SetFilePointer(handle1, (DWORD) pos, 1 + (DWORD *)&pos, 0);
	if (argc > 2) SetFilePointer(handle2, (DWORD) pos, 1 + (DWORD *)&pos, 0);

	OpenTick = GetTickCount();			// mark time we start reading
	percsec = 0;

	do {								// read and compare loop
		int rdsz = 0;					// assume nothing to read
		__int64 mf = mfsiz - pos;		// bytes left per mfsiz limit
		if (mf >= 0) {					// if something left,
			rdsz = bufsz > mf ? (int) mf : bufsz; // limit to max
		}
		dread(argv[1], handle1, pbf1, rdsz, &chunk1);
		chunkx = chunk2 = chunk1;		// (in case only 1 file)

		if (argc > 2) {					// if 2nd file to process,
			int i;
			dread(pfn2, handle2, pbf2, rdsz, &chunk2);

			chunkx = chunk1 < chunk2 ? chunk1 : chunk2; // min chunk size
			if (chunkx > sskip) {		// if something to compare,
				if (i = memcmp__i32(	// if mismatch,
				  pbf1 + sskip, pbf2 + sskip, chunkx - sskip)) {
					char nbuf[40];
					i = chunkx - i;
					wsprintf(wspbuf, 
						"'%s' & '%s' differ at offset %s: %#2x vs %#2x\n",
						argv[1], pfn2, nfmt(&nbuf[sizeof nbuf], pos + i),
						pbf1[i] & 0xff, pbf2[i] & 0xff);
					fputs(wspbuf, stdout);
					MYEXIT(25);
				}
			}
			if ((sskip -= chunkx) < 0) sskip = 0;
		}

		pos += chunkx;					// update file position
		do {
			DWORD tc = GetTickCount();
			int i = (int) ((tc - OpenTick) / 1000);
			char *pmsg = wspbuf;
			char nbuf[40];
			if (chunkx == bufsz) {		// if haven't hit EOF
				if (i <= percsec || !isatty(fileno(stdout))) break;
			}
			percsec = i;
			i = maxb ? (int) ((pos * 1000) / maxb) : 1000;
			pmsg += wsprintf(pmsg, "%s: %s (%d.%d%%", 
				argc > 2 ? "Compared" : "Read",
				nfmt(&nbuf[sizeof nbuf], pos), i / 10, i % 10);
			if (i == 1000) { lstrcpy(pmsg -= 6, "100% "); pmsg += 5; }
			if (percsec >= 60) pmsg += wsprintf(pmsg, " %d min", percsec/60);
			pmsg += wsprintf(pmsg, "%3d sec)", percsec%60);
			if (dispf) {
				wsprintf(nbuf, " %%.%ds", dispf);
				pmsg += wsprintf(pmsg, nbuf, pfnx);
			}
			*pmsg++ = chunkx != bufsz ? '\n' : '\r', *pmsg = '\0';
			fputs(wspbuf, stdout);
		} while (0);
		if (chunk1 != chunk2) {			// if files are different sizes
			wsprintf(wspbuf, 
				"'%s' & '%s' are different sizes\n", argv[1], pfn2);
			fputs(wspbuf, stdout);
			MYEXIT(30);
		}
	} while (chunkx == bufsz);			// continue until no more data
}
