/*
	phcomp.c - Compares files w/progress timer.  PaulHoule.com

	compiled MSVC 10:
		cl	/Oxsb0 /Gr /GS- phcomp.c /link /filealign:512 /safeseh:no /fixed
			/ignore:4078,4254 /SECTION:.text,EWR /merge:.data=.text
			/merge:.rdata=.text /subsytem:console /entry:main
			kernel32.lib user32.lib shell32.lib
		(include=...;msvc\sdk\include  lib=...;msvc\sdk\lib)

	Change Log:
		11/21/2014 Ver 1.5:	Added -o option (set starting file offset).
							Increased default bufsz x 8 (better speed).

		11/22/2014 Ver 1.6:	Allowed -o to be non-ALGN value (start anywhere).
							Added -m option (ending offset value).
							Replaced some C stdlib calls with equiv. WIN32.

		 1/ 6/2015 Ver 1.7:	Added -d option.
		 					Allowed 2nd filespec to be a path.
							Allowed option numbers to contain ignored commas.
							Fixed -o bug if offset was > filesize.
							Replaced some C stdlib calls with equiv. WIN32.

		 1/20/2015 Ver 1.9:	Allowed empty -m to compare to smallest filesize.
							Fixed bug not returning 0 exitcode on match.

		 2/11/2015 Ver 2.0:	Shrunk .exe to 4k by removing C stdlib.
*/

#include <windows.h>

#define ALGN 4096				// buffer alignment, and minimum buffer size
#define ALGN_DN(x) ((x) & ~(ALGN-1))
#define ALGN_UP(x) ALGN_DN((x) + (ALGN-1))
#define U64(i) ((unsigned __int64)(i))
#define HI32(i) ( ((__int32 *)&(i))[1] )
#define LO32(i) ( ((__int32 *)&(i))[0] )

typedef struct {						// file info structure
	char *fn;							// file name
	HANDLE h;							// handle
	unsigned char *buf;					// I/O buffer
    unsigned ck;						// size of last chunk read
	__int64 fsiz;						// file size
} FINFO;

typedef struct {						// global variables
	char *wspbuf;						// wsprintf() formatting buffer
	char *pfnx;							// ptr to 1st file name, beyond path
	DWORD OpenTick;						// GetTickCount() after file open(s)
	int esec;							// currently displayed elapsed secs
	int skp;							// remaining skip (for -o support)
	int dspf;							// filename display control
	unsigned bufsz;						// file buffer size
	FINFO f1, f2;						// file info
	__int64 pos;						// current position in both files
	__int64 endp;						// file position endpoint
	__int64 mval;						// -m option value (-1 if none)
	char nbuf[32];						// secondary formatting buffer
} GVARS;

/* routines to write to std handles w/o using C standard library */
int wfsh(int nh, char *str) {
	static struct { DWORD nh; HANDLE h; } hlast;
	if (nh != (int) hlast.nh) hlast.h = GetStdHandle(hlast.nh = nh);
	if (!str) return GetFileType(hlast.h) == FILE_TYPE_CHAR;
	do {
		char *es = str - 1;
		do { es++; } while (*es & (char)~'\n' || *es && *es != '\n');
		if (es - str) WriteFile(hlast.h, str, es - str, &nh, NULL);
		if (!*(str = es)) break; // dbl buff to reduce WriteFile calls?
		WriteFile(hlast.h, "\r\n", 2, &nh, NULL);
	} while (*++str);
	return 0;
}
int pso(char *str) { return wfsh(STD_OUTPUT_HANDLE, str); }
int pse(char *str) { return wfsh(STD_ERROR_HANDLE,  str); }

unsigned slen(char *ps) {				// implement strlen()
	char *pss = ps;
	while (*ps++) ;
	return (ps - 1) - pss;
}
char *scpy(char *pd, char *ps) {		// implement strcpy()
	char *pds = pd;
	while (*pd++ = *ps++) ;
	return pds;
}

void *gmem(unsigned bytes) {
	void *pmem = HeapAlloc(GetProcessHeap(), 0, bytes);
	if (!pmem) { pse("Mem Alloc error"); ExitProcess(20); }
	return pmem;
}

char **GetArgs(int *pargc) { // returns argv, or NULL if error
	char **argv = NULL;
	void **wargv = (void **) CommandLineToArgvW(GetCommandLineW(), pargc);
	int i, j = 0;
	if (wargv) {
		while (1) {
			char *pnxt = (char *) &argv[*pargc + 1];
			for (i = 0; i < *pargc; i++) {
				if (argv) argv[i] = pnxt;
				pnxt += WideCharToMultiByte(CP_ACP, 0, wargv[i], -1,
					pnxt, j, NULL, NULL);
			}
			if (argv) break;
			argv = (char **) gmem(j = pnxt - (char *) NULL);
			argv[*pargc] = NULL;
		}
		/* LocalFree(wargv); */			// gets freed on process exit
	}
	return argv;
}

void derror(GVARS *pg, char *fn, char *fmsg, int xcode) {
	wsprintf(pg->wspbuf, "*** Error %d %s '%s'\n", GetLastError(), fmsg, fn);
	pse(pg->wspbuf);
	ExitProcess(xcode);
}

void dopen(GVARS *pg, FINFO *pf) {
	do {								// start do {} while(0) dummy block
		pf->h = CreateFile(pf->fn,		// file name
			GENERIC_READ,				// open for reading
			FILE_SHARE_READ,			// share for reading
			NULL,						// default security
			OPEN_EXISTING,				// existing file only
			FILE_FLAG_NO_BUFFERING,		// don't use/alter Windows disk cache
			NULL);						// no attr. template
		if (pf->h == INVALID_HANDLE_VALUE) break;

		/* can't use SetFileSize() from end to get size due to NO_BUFFERING */
		LO32(pf->fsiz) = GetFileSize(pf->h, &HI32(pf->fsiz));
		if (HI32(pf->fsiz) == INVALID_FILE_SIZE) break;

		SetFilePointer(pf->h, LO32(pg->pos), &HI32(pg->pos), FILE_BEGIN);
		if (HI32(pg->pos) == INVALID_SET_FILE_POINTER) break;

		if (pg->endp > pf->fsiz) pg->endp = pf->fsiz;

		pf->buf = (void *) 				// +16 needed for beyond-EODATA work
			ALGN_UP( (char *)gmem(pg->bufsz + 16 + (ALGN-1)) - (char *)0 );
		return;
	} while (0);
	derror(pg, pf->fn, "opening", 10);
}

void dread(GVARS *pg, FINFO *pf) {
	unsigned rdsz = pf->ck;
	int res = ReadFile(pf->h, pf->buf, ALGN_UP(rdsz), &pf->ck, NULL);
	if (!res || pf->ck < rdsz) derror(pg, pf->fn, "reading", 12);
	pf->ck = rdsz;						// limit pf->ck to rdsz
}

char *nfmt(GVARS *pg, unsigned __int64 n) {
	char *pBuf = pg->nbuf;
	char *praw = pBuf + wsprintf(pBuf, "%I64u", n);
	char cc = 5;
	char *p2 = pBuf + sizeof(pg->nbuf) - 1;
	do {
		if (0 == --cc) *p2-- = ',', cc = 3;
		*p2-- = *praw--;
	} while (praw >= pBuf);
	return p2 + 1;
}

void BlkCmp(GVARS *pg) {
	int nbyt;
	unsigned char *p1 = pg->f1.buf, *p2 = pg->f2.buf;

	if ((nbyt = pg->skp) > 0) {			// handle -o startup alignment skip
		do { *p1++ = *p2++; } while (--nbyt);
		p1 = pg->f1.buf, p2 = pg->f2.buf;
		pg->skp -= pg->f1.ck;
	}

	p1[pg->f1.ck] = 1 + (p2[pg->f1.ck] = 0); // insure terminating mismatch
										// compare data in blocks for speed
	#define BCPO(ptr, off) *(__int32 *)((ptr) + (off))
	#define BCPX(off) BCPO(p1, off) == BCPO(p2, off)
	while (BCPX(0) && BCPX(4) && BCPX(8) && BCPX(12)) p1 += 16, p2 += 16;
	while (*p1 == *p2) p1++, p2++;		// advance to mismatch byte

	nbyt = p1 - pg->f1.buf;
	if (nbyt == pg->f1.ck) return;		// ok if matched all bytes

	wsprintf(pg->wspbuf, "'%s' & '%s' differ at offset %s: %#2x vs %#2x\n",
		pg->f1.fn, pg->f2.fn, nfmt(pg, pg->pos + nbyt), *p1, *p2);
	pso(pg->wspbuf);
	ExitProcess(25);
	return;
}

/*	Use of mul64by32(), and the weird way it's written, is to cause
	Visual C compilers not to link in __allmul().	*/
unsigned __int64 mul64by32(unsigned __int64 u64, unsigned u32) {
	unsigned __int64 r = U64((unsigned) u64) * u32;
	unsigned __int64 h = U64(HI32(u64) * u32) << 32;
	if (HI32(h)) r += h;
	return r;
}

__int64 toi(char *popt) {				// result negative if error
	__int64 r = 0;
	int nx;
	while (nx = *popt++) {
		if (nx == ',') continue;
		if ((nx -= '0') < 0 || nx > 9 || r > ((U64(1)<<63)-1)/10) {
			r = -1; break; }
		r = mul64by32(r, 10) + (unsigned) nx;
	}
	return r;
}

char *ffn(char *pfn) {					// isolate file name (skip over path)
	char *pfe = pfn;
	while (*pfe++) ;
	while (--pfe > pfn && pfe[-1] != '\\' && pfe[-1] != ':') ;
	return pfe;
}

void GiveHelp(void) {
	pso("Compares files w/timer.  Ver 2.0, 2/11/2015 PaulHoule.com\n"
		"If second file omitted, first file is read and discarded.\n\n"
		"Bypasses operating system cache to produce consistent timings\n"
		"and a true test of the underlying storage medium.\n"
		"\n  Usage:  phcomp {-opt(s)} file1 {file2|path}\n\n"
		"-?    This display\n"
		"-d#   Display # chars of file name\n"
		"-b#   Read buffer size (4k min, default 4 meg)\n"
		"-o#   Start file offset\n"
		"-m#   Max file offset (-m for smallest filesize)\n"
	);
	ExitProcess(1);
}

void do_opt(GVARS *pg, char *popt) {
	#define DO_OFF(fld) (unsigned) &((GVARS *)0)->fld
	static char odef[] = {
		'o',DO_OFF(pos)+0x80,	'd',DO_OFF(dspf),
		'm',DO_OFF(mval)+0x80,	'b',DO_OFF(bufsz), 0
	};
	unsigned char *pod = odef;

	if (popt[1] == '?') GiveHelp();
	do {
		if (*pod++ == (unsigned char) (popt[1] | 0x20)) {
			__int64 v = toi(popt + 2);
			unsigned o = *pod & 0x7f;
			*(__int32 *) ((char *)pg+o) = LO32(v);
			if (*pod != (unsigned char) o) {
				*(__int64 *) ((char *)pg+o) = v;
				LO32(v) = HI32(v);
			}
			if (LO32(v) < 0) break;
			return;
		}
	} while (*++pod);
	wsprintf(pg->wspbuf, "bad opt: %s\n", popt);
	pse(pg->wspbuf);
	ExitProcess(99);
}

void do_opts(GVARS *pg, char **argv) {
	char *arg, **wargv = argv;
	while (arg = *wargv = *argv++) {
		if (*arg == '-' || *arg == '/') do_opt(pg, arg);
		else wargv++;
	}
}

unsigned calcper(unsigned __int64 num, unsigned __int64 denom) {
										// scale down to avoid 64-bit divide
	while (denom & ~U64(0x3fffff)) num >>= 1, denom >>= 1;
	return (int) denom ? (unsigned) num * 1000u / (unsigned) denom : 1000;
}

void UpdMsg(GVARS *pg) {
	int i = (int) ((GetTickCount() - pg->OpenTick) / 1000);
	char *pmsg = pg->wspbuf;

	if (pg->f1.ck) {					// if not final UpdMsg() call
		if (i <= pg->esec) return;		// wait till a second has passed
		if (!pso(NULL)) return;			// no msg if output is to a file
	}
	pg->esec = i;
	i = calcper(pg->pos, pg->endp);
	pmsg += wsprintf(pmsg, "%s: %s (%d.%d%%", pg->f2.fn ? "Compared" : "Read",
		nfmt(pg, pg->pos), i / 10, i % 10);
	if (i == 1000) { scpy(pmsg -= 6, "100% "); pmsg += 5; }
	if (pg->esec >= 60) pmsg += wsprintf(pmsg, " %d min", pg->esec / 60);
	pmsg += wsprintf(pmsg, "%3d sec)", pg->esec % 60);
	if (pg->dspf >= 0) {
		wsprintf(pg->nbuf, " %%.%ds", pg->dspf ? pg->dspf : 44);
		pmsg += wsprintf(pmsg, pg->nbuf, pg->pfnx);
	}
	*pmsg++ = pg->f1.ck ? '\r' : '\n', *pmsg = '\0';
	pso(pg->wspbuf);
}

__cdecl main(int argc, char *argv[]) {
	GVARS g;							// "global" variables

	g.bufsz = 1024 * ALGN;				// default I/O buffer size
	g.pos = 0;							// current position in both files
	g.esec = 0;							// elapsed seconds
	g.dspf = -1;						// init to no filename display
	g.mval = -1;						// positive when -m specified
	g.wspbuf = gmem(1040);				// wsprintf() formatting buffer

	argv = GetArgs(&argc);				// (note: we don't use/update argc)
	do_opts(&g, &argv[1]);				// parse/remove any options
	if (!argv[1]) GiveHelp();			// give help if no filename

	g.bufsz = ALGN_DN(g.bufsz);			// insure bufsz aligned and non-zero
	if (!g.bufsz) g.bufsz = ALGN;
	LO32(g.pos) -= g.skp = (int) g.pos & (ALGN-1); // align pos
	g.endp = g.mval;					// set default ending offset
	if (g.mval <= 0) HI32(g.endp) = 0x7fffffff;

	g.pfnx = ffn(g.f1.fn = argv[1]);	// isolate file name (beyond any path)
	dopen(&g, &g.f1);					// open 1st file
	if (g.f2.fn = argv[2]) {			// open 2nd file, if there
		DWORD fa = GetFileAttributes(g.f2.fn);
		if (fa != -1 && fa & FILE_ATTRIBUTE_DIRECTORY) {
			int dl = slen(g.f2.fn);
			g.f2.fn = scpy(gmem(dl + 1 + slen(g.pfnx) + 1), g.f2.fn);
			if (g.f2.fn[dl - 1] != ':' && g.f2.fn[dl - 1] != '\\')
				g.f2.fn[dl++] = '\\';
			scpy(g.f2.fn + dl, g.pfnx);
		}
		dopen(&g, &g.f2);
	}
	if (g.pos > g.endp) g.pos = g.endp;

	g.OpenTick = GetTickCount();		// mark time we start reading

	do {								// read and compare loop
		g.f1.ck = g.bufsz;
		if (g.f1.ck > g.endp - g.pos) g.f1.ck = (unsigned) (g.endp - g.pos);
		dread(&g, &g.f1);

		if (g.f2.fn) {					// if 2nd file to read and compare,
			g.f2.ck = g.f1.ck;
			dread(&g, &g.f2);
			BlkCmp(&g);
		}

		g.pos += g.f1.ck;				// update file position
		UpdMsg(&g);						// update progress message
	} while (g.f1.ck);

	if (g.f2.fn && HI32(g.mval) < 0 && g.f1.fsiz != g.f2.fsiz) {
		wsprintf(g.wspbuf, "'%s' & '%s' are different sizes\n",
			g.f1.fn, g.f2.fn);
		pso(g.wspbuf);
		ExitProcess(30);
	}
	ExitProcess(0);
}
