/*
	phima.c - Create .IMA file for specified drive.
	Copyright (c) 2006, Paul Houle (http://paulhoule.com).
	All rights reserved.

	This is a 16-bit DOS program to create a .IMA file, typically
	of a source floppy.  The extended INT 25h interface is used to
	access the drive data, or the newer INT 21h func 7305h if the DOS
	version is from Win 95 OSR2 or later.

	Images of large FAT32 drives can also be created - if they exceed
	the DOS filesize limit, .001 .002 etc. extension files are created.

	Compiled w/MSFT C 1.52 via "cl /AT /Ozaxs phima.c"
	then compressed w/UPX (http://upx.sourceforge.net/)
*/

#include <stdio.h>
#include <stdlib.h>
#include <io.h>
#include <fcntl.h>
#include <string.h>

char ofname[_MAX_PATH];					// output file name
FILE *ofhndl = NULL;					// output file handle (NULL if closed)

// When the image source is a boot CD simulated floppy (which is why I wrote
// this), xferring multiple sectors is over 10x faster (total execution time)
// than xferring 1.  Same is true for a real floppy drive... evidently single
// sector transfers can cause a rotational latency on these devices.

char secbuf[64u * 512u];				// sector buffer

int ldrv = 0;							// default source drive (0 = A, etc)
unsigned long lsec = 0;					// next logical number to read
unsigned long tsec = 0;					// total logical source sectors

void GiveHelp(void);
void MakeFn(char *fn, int force, char *ext); // make filename (adds .ext)
void OpenOutfile(void);					// open{/create} output file
void WriteOutfile(char *pd, unsigned dsiz);	// write data to output file
void CloseOutfile(void);				// close output file
void GetLD(char *pld);					// get source drive from command line
void EvalLD(void);						// evaluate souce drive
int ReadSectors(void);					// read block of sectors
void GetSec(int ldrv, unsigned long lsec, int nsec, char *buff); // low-level
void ProgressMsg();						// display progress message

void main(int argc, char **argv) {

	if (argc < 2) GiveHelp();			// dump help if not enough args
	if (argc > 2) GetLD(argv[2]);		// get source drive
	EvalLD();							// evaluate drive (test, get size)

	MakeFn(argv[1], 0, "ima");			// make first outfile name
	OpenOutfile();						// open output file

	while (lsec < tsec) {				// while sectors remain,
		int ns = ReadSectors();			// read next block of sectors
		lsec += ns;						// update sector position
		WriteOutfile(secbuf, ns * 512);	// write the sector data
		ProgressMsg();					// display progress message
	}

	CloseOutfile();
}

void OpenOutfile(void) {				// open output file
	if (ofname[0] == '-') {				// if asking for stdout,
		ofhndl = stdout;				// pipe output
		setmode(fileno(ofhndl), O_BINARY);	// set binary mode
	} else {
		ofhndl = fopen(ofname, "wb");
	}
	if (ofhndl == NULL) {
		fprintf(stderr, "\nError opening output: %s\n", ofname);
		exit(-1);
	}
}

void WriteOutfile(char *pd, unsigned dsiz) { // write data to output file
	#define MFSIZE 2147483647l			// Max allowed file size
	static long ThisChunk = MFSIZE;		// chunk filesize downcounter
	static int cnum = 0;				// current chunk file number (.nnn)

	while ((ThisChunk -= dsiz) < 0) {	// if current file would overflow,
		char ext[4];
		CloseOutfile();					// close current file
		sprintf(ext, "%03d", ++cnum);	// format next extension
		MakeFn(ofname, 1, ext);			// make new .nnn outfile name
		OpenOutfile();					// open the new output file
		ThisChunk = MFSIZE;				// reset filesize downcounter
	}

	if (fwrite(pd, 1, dsiz, ofhndl) != dsiz) { // write data
		fprintf(stderr, "\nError writing %s at sector %lu\n", ofname, lsec);
		exit(-1);
	}
}

void CloseOutfile(void) {				// close output file
	if (ofhndl) {						// if a file is open,
		if (ofhndl != stdout) {			// don't close if stdout
			if (fclose(ofhndl)) {		// close output file
				fprintf(stderr, "\nError closing %s\n", ofname);
				exit(-1);
			}
		}
		ofhndl = (FILE *) NULL;
	}
}

void MakeFn(							// Make filename (add .ext if needed)
	char *fn,							// Source file name (can't be NULL)
	int force,							// if non-zero, remove existing .ext
	char *ext							// our "xxx" extension (1..3 chars)
) {
	char *onm = ofname;					// start of output buffer
	char *px = NULL;					// ptr to .ext in output, NULL if none

	do {								// copy source fn to output buffer
		if (*fn == '\0') break;			// stop when terminator hit
		if (*fn == '.') px = onm;		// check for .ext as we copy
		if (*fn == '\\') px = NULL;
		*onm++ = *fn++;
	} while (onm < &ofname[sizeof(ofname) - 5]); // (reserve 5 for ".ext",0)

	if (force && px) {					// if stripping existing .ext, do so
		onm = px;
		px = NULL;
	}

	if (!px && ofname[0] != '-') {		// if no .ext in fn, add one
		*onm++ = '.';
		while ((*onm++ = *ext++) != 0) ;
	} else {							// else just add terminator
		*onm = '\0';
	}
}

void GiveHelp(void) {
	printf(
		"\nVer 1.0 - Make .ima file.  (c) 2006, PaulHoule.com.\n"
		"Makes drive image file, suitable for burning "
		"a bootable CD.  Typically\n"
		"for a floppy, but will work with FAT32, splitting output"
		" across files.\n\n"
		"    USAGE: phima Outfile{.ima} {Drive{:}}\n\n"
		"Outfile may be \"-\" (no quotes) for stdout.\n"
		"Drive is A..Z (defaults to A:).\n"
	);
	exit(-1);
}

void GetLD(char *pld) {					// get logical drive number
	ldrv = (*pld & ~0x20) - 'A';		// calc drive (A-Z == 0..25)
	if (ldrv < 0 || ldrv > 25			// if bad drive letter,
	 || (pld[1] && (pld[1] != ':' || pld[2])) // or improper termination
	) {
		fprintf(stderr, "Bad drive letter (A-Z valid): %s\n", pld);
		exit(-1);
	}
}

void EvalLD(void) {						// tests drive, get size in sectors

	GetSec(ldrv, 0, 0, NULL);			// force get DPB (floppy access fix)
	GetSec(ldrv, 0, 1, secbuf);			// get 1st sector

	tsec = *(unsigned *) &secbuf[0x13];	// get short sec count from BPB
	if (tsec == 0) {					// if too big,
		tsec = *(unsigned long *) &secbuf[0x20]; // get long sec count
	}

	if (*(int *) &secbuf[0xb] != 512) {	// insure sector size is 512
		fprintf(stderr, "Error -- %c: sector size not 512 (is %d)\n",
			ldrv + 'A', *(int *) &secbuf[0xb]);
		exit(-1);
	}
}

int ReadSectors(void) {					// read next block of sectors
	int ns = sizeof(secbuf) / 512;		// assume we'll read max sectors
	unsigned long ts = tsec - lsec;		// total sectors that remain

	if ((unsigned) ns > ts) ns = (int) ts; // limit xfer to what remains
	GetSec(ldrv, lsec, ns, secbuf);		// get the sectors
	return ns;
}

/*	Note: this uses extended int 25h calls, which first appeared in some
	versions of DOS 3.3 (e.g. Compaq?), then completely in DOS 4.0.
	Or, int 21h func 7305h is used, if the version of DOS is recent enough.
	The max drive supported is 2 terabytes (2*32 x 512).
*/
#pragma warning(disable:4704)
void GetSec(							// get logical secs
	int ldrv,							// logical drive (0..25 == A..Z)
	unsigned long lsec,					// logical sector (0..n)
	int nsec,							// sectors requested
	char *buff							// buffer for data
) {
	static int ver = -1;
	int rc;
	_asm {
		push	ds
		push	bp
		push	si
		push	di

		mov		ax,ver					// get DOS version
		cmp		ax,-1					// have got it yet (only done once)?
		jne		gotver					// jmp= yes

		mov		ah,30h					// get DOS version
		int		21h
		xchg	ah,al					// ah= major version, al= minor
		mov		ver,ax					// save for subsequent calls

	gotver:
		mov		dx,ldrv					// dx= drive number (0= A, 1= B, etc)
		mov		cx,-1					// # secs= 0xffff for param struc

		cmp		nsec,0					// **** kludge to fix a floppy
		jne		dord					//	problem -- if #secs to transfer
		inc		dx						//	is zero, we do a DOS GET DPB.
		mov		ah,32h					//	This should be done first so
		int		21h						//	DOS will initialize its media
		xor		ax,ax					//	geometry.
		jmp short donegd
	dord:

		push	ds						// build DISKIO structure on stack
		push	word ptr buff
		push	nsec
		push	word ptr lsec+2
		push	word ptr lsec
		push	ss						// ds:bx= ptr to DISKIO struc
		pop		ds
		mov		bx,sp

		cmp		ax,700h					// version GREATER THAN 7.0?
		ja		use7305					// jmp= yes, use new-style call
										// not at least Win 95 OSR2
		xchg	ax,dx					// ax= drive number
		int		25h						// old-style disk sector read
		pop		dx						// discard flags DOS leaves on stack
		jmp short donerd

	use7305:							// Win 95 OSR2 or later
		inc		dx						// bias drive number (1= A, 2= B, etc)
		xor		si,si					// function is read
		mov		ax,7305h				// extended (FAT32-capable) I/O call
		int		21h						// new-style disk sector read

	donerd:
		sbb		ah,ah					// ah= -1 if error, else 0
		add		sp,5 * 2				// remove DISKIO structure
		and		al,ah					// zero al unless ah indicates error
	donegd:
		pop		di
		pop		si
		pop		bp
		pop		ds
		mov		rc,ax					// (pass rc back to C code)
	}
	if (rc) {
		fprintf(stderr, "\nError %d reading drive %c: at sector %lu\n",
			rc & 0xff, ldrv + 'A', lsec);
		exit(-1);
	}
}

void ProgressMsg() {					// display a progress message
	static int lper = -1;				// previous percent completed
	static unsigned long lmsc = 0;		// amount of data at last message
	int per;							// current percent completed
	char fbuf[20];						// number format buffer
	char *ard, *awr;					// work ptr
	int cnt;

	if (ofname[0] == '-') return;		// no msg if dumping to stdout

	// Generate percentage complete (per) and decide to display it.

	per = 100;							// assume 100% complete
	if (lsec < tsec) {					// if assumption wrong, calc percent
		unsigned bias = tsec >= 0x2000000 ? 128 : 1; // (overflow prevention)
		per = (int) (100 * (lsec / bias) / (tsec / bias));

		// If size over a gig, output every 10 meg; otherwise output every
		// percent.  This gives reasonable speed updates for > 1 gig images.

		if (tsec >= 0x200000) {			// if total size a gig or more,
			if (lsec - lmsc <= 0x5000) return; // exit if not 10 meg yet
			lmsc = lsec;				// mark this spot and perform display
		} else {
			if (lper == per) return;	// exit if percentage hasn't changed
			lper = per;					// mark last percent displayed
		}
	}

	// Format the sector count into a temp buffer.
	// The subsequent block multiplies sectors by 512 in the char domain
	// yielding byte count (done this way since it might not fit in a long).

	sprintf(fbuf, "%15lu", lsec);		// (allow for 14 active digits below)
	cnt = 9;
	do {
		int car = '0';
		awr = &fbuf[14];	// last digit
		do {
			*awr = (char) ((*awr & 0xf) * 2 + car);
			car = '0';
			if (*awr > '9') {
				*awr -= 10;
				car = '1';
			}
		} while (*--awr != ' ' || car != '0');
	} while (--cnt);

	// Insert comma separators to make big values easier to understand.

	cnt = 5;							// tells us when to insert a ','
	ard = &fbuf[15];					// 0 terminator of source digits
	awr = &fbuf[sizeof(fbuf) - 1];		// new destination for 0 terminator
	do {
		if (!--cnt) {					// if time to insert a ','
			*awr-- = ',';				// do so
			cnt += 3;
		}
		*awr-- = *ard--;
	} while (*ard != ' ');				// continue till out of source digits
	awr++;								// point to start of formatted number

	printf("\rBytes written: %s (%d%%)", awr, per);
	if (per == 100) printf("\n");
}
