/*
	phima.c - Create .IMA/.ISO file for specified drive.
	Copyright (c) 2006, Paul Houle (http://paulhoule.com/phima).
	All rights reserved.

	This is a simple 16-bit DOS command-line program to create an image
	file, typically of a source floppy, though images of FAT32 volumes
	and CD's or DVD's (MSCDEX devices) are also supported in 16-bit DOS
	(Windows disallows direct DOS device access).  Note: a CD/DVD image
	is actually an ISO file, and may be named .ISO and used as such.

	Optionally the start sector and number of sectors can be specified.
	Example:  'phima boot A: 0 1'  dumps the boot sector of drive A:

	Large outputs can be generated -- if the output exceeds the 2gb DOS
	filesize limit, .001 .002 etc. extension files are created.

	Compiled w/MSFT C 1.52 via "cl /AT /G3 /Ozaxs phima.c"
	then compressed w/UPX (http://upx.sourceforge.net/)
*/

#include <stdio.h>
#include <stdlib.h>
#include <io.h>
#include <fcntl.h>
#include <string.h>
#pragma warning(disable:4704)

char ofname[_MAX_PATH];					// output file name
FILE *ofhndl = NULL;					// output file handle (NULL if closed)

// When the image source is a boot CD simulated floppy (which is why I wrote
// this), xferring multiple sectors is over 10x faster (total execution time)
// than xferring 1.  Same is true for a real floppy drive... evidently single
// sector transfers can cause a rotational latency on these devices.

unsigned char secbuf[32768u];			// sector buffer (sector size varies)

int ldrv = 0;							// default source drive (0 = A, etc)
int iotype = 0;							// type of I/O used (see EvalLD fn)
unsigned long ssec = 0;					// starting sector of dump
unsigned long esec = 0;					// ending   sector of dump
unsigned long tsec = 0;					// total source volume sectors
unsigned long lsec = 0;					// next sector number to read
unsigned secsize = 0;					// sector size in bytes

unsigned char cdb[32];					// used to build small control blocks
struct {								// used for doint() call only
	unsigned ax; unsigned bx; unsigned cx; unsigned dx;
	unsigned si; unsigned di; unsigned ds; unsigned es; unsigned cf;
} regs;
void doint(int inum);					// execute x86 interrupt

void GiveHelp(void);
void MakeFn(char *fn, int force, char *ext); // make filename (adds .ext)
void OpenOutfile(void);					// open{/create} output file
void WriteOutfile(void *pd, unsigned dsiz);	// write data to output file
void CloseOutfile(void);				// close output file
void GetLD(char *pld);					// get source drive from command line
void EvalLD(void);						// evaluate source drive
void SetRange(int argc, char **argv);	// set specific sector range to dump
int ReadSectors(void);					// read block of sectors
void GetSec(int ldrv, unsigned long lsec, int nsec, void *buff); // low-level
char *NFmt(char *pBuf, unsigned long nm, unsigned sc2); // format big number
void ProgressMsg();						// display progress message

void main(int argc, char **argv) {

	if (argc < 2) GiveHelp();			// dump help if not enough args
	if (argc > 2) GetLD(argv[2]);		// get source drive letter

	EvalLD();							// evaluate drive (test, get size)

	lsec = 0;							// assume we'll dump entire volume
	esec = tsec;
	if (argc > 3) SetRange(argc, argv);	// allow selection of sector range

	MakeFn(argv[1], 0, "ima");			// make outfile name
	ProgressMsg();						// dump 1st stats line (total bytes)
	if (ofname[0] == '?') return;		// give stats only if '?' filename
	OpenOutfile();						// open output file

	while (lsec < esec) {				// while sectors remain,
		int ns = ReadSectors();			// read next block of sectors
		lsec += ns;						// update sector position
		WriteOutfile(secbuf, ns * secsize);	// write the sector data
		ProgressMsg();					// display running progress message
	}

	CloseOutfile();
	exit(0);
}

void SetRange(int argc, char **argv) {	// select sector range to dump
	char *pec;

	lsec = ssec = strtoul(argv[3],  &pec, 0);	// set start sector number
	if (!*pec && ssec <= tsec) {		// if no error,
		esec = tsec;					// assume "all" sectors
		if (argc <= 4) return;
		esec = strtoul(argv[4],  &pec, 0);	// get sector count
		if (!*pec && esec <= tsec - ssec) {	// if no error,
			esec += ssec;				// convert to ending sector
			return;
		}
	}
	fprintf(stderr, "Illegal start/count format or value\n");
	exit(-1);
}

void OpenOutfile(void) {				// open output file
	if (ofname[0] == '-') {				// if asking for stdout,
		ofhndl = stdout;				// pipe output
		setmode(fileno(ofhndl), O_BINARY);	// set binary mode
	} else {
		ofhndl = fopen(ofname, "wb");
	}
	if (ofhndl == NULL) {
		fprintf(stderr, "\nError opening output: %s\n", ofname);
		exit(-1);
	}
}

void WriteOutfile(void *pd, unsigned dsiz) { // write data to output file
	#define MFSIZE 2147483647l			// Max allowed file size
	static long ThisChunk = MFSIZE;		// chunk filesize downcounter
	static int cnum = 0;				// current chunk file number (.nnn)

	while ((ThisChunk -= dsiz) < 0) {	// if current file would overflow,
		char ext[4];
		CloseOutfile();					// close current file
		sprintf(ext, "%03d", ++cnum);	// format next extension
		MakeFn(ofname, 1, ext);			// make new .nnn outfile name
		OpenOutfile();					// open the new output file
		ThisChunk = MFSIZE;				// reset filesize downcounter
	}

	if (fwrite(pd, 1, dsiz, ofhndl) != dsiz) { // write data
		fprintf(stderr, "\nError writing %s at sector %lu\n", ofname, lsec);
		exit(-1);
	}
}

void CloseOutfile(void) {				// close output file
	if (ofhndl) {						// if a file is open,
		if (ofhndl != stdout) {			// don't close if stdout
			if (fclose(ofhndl)) {		// close output file
				fprintf(stderr, "\nError closing %s\n", ofname);
				exit(-1);
			}
		}
		ofhndl = NULL;
	}
}

void MakeFn(							// Make filename (add .ext if needed)
	char *fn,							// Source file name (can't be NULL)
	int force,							// if non-zero, remove existing .ext
	char *ext							// our "xxx" extension (1..3 chars)
) {
	char *onm = ofname;					// start of output buffer
	char *px = NULL;					// ptr to .ext in output, NULL if none

	do {								// copy source fn to output buffer
		if (*fn == '\0') break;			// stop when terminator hit
		if (*fn == '.') px = onm;		// check for .ext as we copy
		if (*fn == '\\') px = NULL;
		*onm++ = *fn++;
	} while (onm < &ofname[sizeof(ofname) - 5]); // (reserve 5 for ".ext",0)

	if (force && px) {					// if stripping existing .ext, do so
		onm = px;
		px = NULL;
	}

	if (!px && ofname[0] != '-') {		// if no .ext in fn, add one
		*onm++ = '.';
		while ((*onm++ = *ext++) != 0) ;
	} else {							// else just add terminator
		*onm = '\0';
	}
}

void GiveHelp(void) {
	printf(
		"\nVer 2.0 - Make .ima/.iso file.  (c) 2006, PaulHoule.com.\n"
		"Makes drive image file.  Originally for a floppy, but will work\n"
		"with CD/DVD/FAT32, splitting output across files if necessary.\n"
		"Note: only floppies can be imaged in a DOS box -- use true DOS.\n\n"
		"    USAGE: phima Outfile{.ima} {Drive{:} {Start {Count}}}\n\n"
		"Outfile may be - for stdout, or ? for stats only.\n"
		"Drive is A..Z (defaults to A:).\n"
		"Start is start sector number (defaults to 0).\n"
		"Count is # sectors (defaults to all).\n"
	);
	exit(-1);
}

void GetLD(char *pld) {					// get logical drive number
	ldrv = (*pld & ~0x20) - 'A';		// calc drive (A-Z == 0..25)
	if (ldrv < 0 || ldrv > 25			// if bad drive letter,
	 || (pld[1] && (pld[1] != ':' || pld[2])) // or improper termination
	) {
		fprintf(stderr, "Bad drive letter (A-Z valid): %s\n", pld);
		exit(-1);
	}
}

void CheckCD(void) {					// test for CD -- if so, get its size
	regs.ax = 0x150b;					// test if the logical drive (ldrv)
	regs.bx = 0;						//	is an MSCDEX controlled device
	regs.cx = ldrv;
	doint(0x2f);

	if (regs.bx == 0xadad && regs.ax) {	// If MSCDEX device,
		iotype = 1;						// set MSCDEX iotype and sector size
		secsize = 2048;

		memset(cdb, 0, sizeof(cdb));	// init (zero) command buffer
		cdb[0] = 26;					// format IOCTL call header:
		cdb[2] = 3;						//	command code (3 for IOCTL input)
		*(void _far **) &cdb[14] = &cdb[26]; // ptr to IOCTL control block
		cdb[26] = 8;					//	"return volume size" command code

		regs.ax = 0x1510;				// execute IOCTL call to get the
		regs.cx = ldrv;					//	total sector count
		regs.bx = (unsigned) cdb;
		regs.es = (_segment) cdb;
		doint(0x2f);

		if (*(unsigned *) &cdb[3] & 0x8000) { // if error,
			fprintf(stderr, "MSCDEX error %d getting volume size for %c:\n",
				*(unsigned *) &cdb[3] & 0xff, ldrv + 'A');
			exit(-1);
		}
		tsec = *(unsigned long *) &cdb[27] - 150; // size (- 150 for pre-gap)
	}
}

void EvalLD(void) {						// tests drive, get size in sectors
	regs.dx = ldrv + 1;					// first perform a "get DPB" (ignoring
	regs.ax = 0x3200;					//	the result) -- this seems to force
	doint(0x21);						//	recognition of media change

	CheckCD();							// check if MSCDEX drive
	if (iotype == 1) return;			// if yes, return (done evaluating)

	regs.ax = 0x3000;					// set iotype to 2 (int 25h I/O) or 3
	doint(0x21);						//	(int 21h func 7305h IO) depending
	iotype = (char) regs.ax >= 7 ? 3 : 2; // on the OS version

	GetSec(ldrv, 0, 1, secbuf);			// get 1st sector

	tsec = *(unsigned *) &secbuf[0x13];	// get short sec count from BPB
	if (tsec == 0) {					// if too big for short field,
		tsec = *(unsigned long *) &secbuf[0x20]; // get long sec count
	}

	secsize = 8;						// fetch/validate sector size
	do {
		if (secsize >= 0x8000) {		// abort if unsupported sector size
			fprintf(stderr, "Error -- %c: invalid sector size (is %d)\n",
				ldrv + 'A', *(unsigned *) &secbuf[0xb]);
			exit(-1);
		}
	} while ((secsize *= 2) != *(unsigned *) &secbuf[0xb]);
}

int ReadSectors(void) {					// read next block of sectors
	int ns = sizeof(secbuf) / secsize;	// assume reading max there's room for
	unsigned long ts = esec - lsec;		// total sectors that remain

	if ((unsigned) ns > ts) ns = (int) ts; // limit xfer to what remains
	GetSec(ldrv, lsec, ns, secbuf);		// get the sectors
	return ns;
}

/*	Get physical sector(s).  This uses either:
	(iotype=1) int 2fh func 1510h, if the source volume is a MSCDEX device.
	(iotype=2) extended int 25h calls, which first appeared in some
		versions of DOS 3.3 (e.g. Compaq?), then completely in DOS 4.0.
	(iotype=3) int 21h func 7305h, if the version of DOS is recent enough.
*/

void GetSec(							// get logical secs
	int ldrv,							// logical drive (0..25 == A..Z)
	unsigned long lsec,					// logical sector (0..n)
	int nsec,							// sectors requested
	void *buff							// buffer for data
) {
	*(unsigned long *) cdb = lsec;		// build/point ds:bx to DISKIO struct
	*(unsigned *) &cdb[4] = nsec;		//	(note: this won't be used -- it's
	*(void _far **) &cdb[6] = buff;		//	overwritten -- for MSCDEX read)
	regs.bx = (unsigned) cdb;
	regs.ds = (_segment) cdb;
	regs.cx = -1;						// (indicate DISKIO being used)

	switch (iotype) {
	  case 1:						// MSCDEX I/O (int 2fh ax=1510h)
		memset(cdb, 0, sizeof(cdb));	// init (zero) req hdr buffer
		cdb[0] = 27;					//	command size
		cdb[2] = 128;					//	command code (READ LONG)
		*(void _far **) &cdb[14] = buff;//	I/O buffer address
		*(int *) &cdb[18] = nsec;		//	number of sectors to xfer
		*(unsigned long *) &cdb[20] = lsec;	//	logical starting sector

		regs.ax = 0x1510;				// READ LONG (HSG addr mode, cooked)
		regs.cx = ldrv;
		regs.bx = (unsigned) cdb;
		regs.es = (_segment) cdb;
		doint(0x2f);
		regs.ax = *(unsigned *) &cdb[3];// al= possible driver error code
		regs.cf = regs.ax & 0x8000;		// regs.cf = non-zero if error
		break;
	  case 2:						// int 25h I/O
		regs.ax = ldrv;
		doint(0x25);
		break;
	  case 3:						// int 21h ax=7305h I/O
		regs.dx = ldrv + 1;
		regs.ax = 0x7305;
		doint(0x21);
		break;
	}
	if (regs.cf) {
		fprintf(stderr,
			"\nError %d reading %d sec(s) of drive %c: at sector %lu\n",
			regs.ax & 0xff, nsec, ldrv + 'A', lsec);
		exit(-1);
	}
}

// Format large (up to 15-digit) numeric nm*sc2 into pBuf, inserting commas.
// pBuf must be at least 20 bytes.  sc2 must be a power of 2 (1 to disable).
// Returns pointer to start of formatted result (zero-terminated).

char *NFmt(char *pBuf, unsigned long nm, unsigned sc2) {
	sprintf(pBuf, "%15lu", nm);			// allow for 15 active digits

	while (sc2 >>= 1) {					// scale ascii number by power-of-2
		int car = '0';
		char *awr = &pBuf[14];			// last formatted digit
		do {
			*awr = (char) ((*awr & 0xf) * 2 + car);
			car = '0';
			if (*awr > '9') {
				*awr -= 10;
				car = '1';
			}
		} while (--awr >= pBuf && (*awr != ' ' || car != '0'));
	}

	{	// Insert comma separators (makes big values easier to read)
		int cnt = 5;					// tells us when to insert a ','
		char *ard = &pBuf[15];			// 0 terminator of source digits
		char *awr = &pBuf[19];			// new destination for 0 terminator
		do {
			if (!--cnt) {				// if time to insert a ','
				*awr-- = ',';			// do so
				cnt += 3;
			}
			*awr-- = *ard--;
		} while (awr >= pBuf && *ard != ' ');
		return awr + 1;					// return start of number
	}
}

void ProgressMsg() {					// display a progress message
	static int tbw = 0;					// width of formatted total bytes
	static int lper = -1;				// previous percent completed
	static unsigned long lmsc = 0;		// amount of data at last message
	int per;							// current percent completed
	char b1[20], b2[20], b3[20], b4[20]; // format buffers
	unsigned long blsec = lsec - ssec;	// lsec,esec biased to ssec offsets
	unsigned long besec = esec - ssec;

	if (ofname[0] == '-') return;		// no msg if dumping to stdout

	if (lper == -1) {					// if 1st time, display stats
		char *ptot = NFmt(b3, besec, secsize); // format total bytes
		tbw = strlen(ptot) + 1;			// set for progress msg
		printf("%c: total bytes: %s (start=%s count=%s sec size=%s)\n",
			ldrv + 'A', ptot, NFmt(b1, ssec, 1), NFmt(b2, besec, 1),
			NFmt(b4, secsize, 1));
		lper--;							// only do this once
		return;
	}

	// Generate percentage complete (per) and decide to display it.

	per = 100;							// assume 100% complete
	if (blsec < besec) {				// if assumption wrong, calc percent
		unsigned bias = besec >= 0x2000000 ? 7 : 0; // (overflow prevention)
		per = (int) (100 * (blsec >> bias) / (besec >> bias));

		// If size over a gig, output every 10 meg; otherwise output every
		// percent.  This gives reasonable speed updates for > 1 gig dumps.

		if (besec >= 0x40000000u / secsize) { // if total >= a gigabyte,
			if (blsec < lmsc + 0xa00000u / secsize) return; // wait on size
			lmsc = blsec;				// mark this spot and perform display
		} else {
			if (lper == per) return;	// exit if percentage hasn't changed
			lper = per;					// mark last percent displayed
		}
	}
	printf("\rBytes written: %*s (%d%%)", tbw, NFmt(b1, blsec, secsize), per);
	if (per == 100) printf("\n");
}

#define LDR(rg) } _asm { mov rg,regs.rg } _asm {
#define SVR(rg) } _asm { mov regs.rg,rg } _asm {

void doint(int inum) {					// exec INT inum using global "regs"
	_asm {
		mov		ax,inum					// form "int inum" opcode
		mov		byte ptr cs:[intxx+1],al
		push	bp						// save key regs
		push	si
		push	di
		push	es
		push	ds
		mov		bp,sp					// save sp (int inum can't modify bp!)
		LDR(ax) LDR(bx) LDR(cx) LDR(dx) LDR(si) LDR(di) LDR(es) LDR(ds)
	intxx:
		int		0						// (note: opcode gets patched above)
		mov		sp,bp					// restore sp (int 25h corrupts it)
		mov		bp,ds
		pop		ds
		mov		regs.ds,bp				// save all regs (and carry)
		SVR(ax) SVR(bx) SVR(cx) SVR(dx) SVR(si) SVR(di) SVR(es)
		sbb		ax,ax
		mov		regs.cf,ax
		pop		es
		pop		di
		pop		si
		pop		bp
	}
}
