/*----------------------------------------------------------------------------
;
; 32-bit X86 MD5 hash generator 1.0 -- Paul Houle (paulhoule.com) 4/12/2010
;
; Non-time critical C logic.  All API entry points are here.
; See phmd5.h for documentation.
;
;---------------------------------------------------------------------------*/

#include <string.h>
#include "phmd5.h"

// Prototype external assembly routine (95%+ of the time is spent there).
extern void Phmd5DoBlocks(unsigned char *hash, char *pdata, unsigned dbytes);

// First call -- initialize pmd5 structure for use.
void Phmd5Begin(PHMD5 *pmd5) {
	unsigned *uhash = (unsigned *) pmd5->hash;

	uhash[0] = 0x67452301;				// init hash per rfc1321
	uhash[1] = 0xEFCDAB89;
	uhash[2] = 0x98BADCFE;
	uhash[3] = 0x10325476;

	pmd5->totbyt[0] = 0;				// init count of data bytes processed
	pmd5->totbyt[1] = 0;
}

// Last call -- after this, pmd5->hash holds final MD5 hash.
void Phmd5End(PHMD5 *pmd5) {
	char pad[72];						// pad buffer (worst case is 72 bytes)
	unsigned padc;						// size of needed pad (9-72 bytes)

	padc = 64 - (pmd5->totbyt[0] & 63);	// assume pad to next 64-byte boundary
	if (padc < 9) padc += 64;			// add a block if we need more room
	memset(pad, 0, padc);				// clear entire pad area
	pad[0] = 0x80;						// place input stream terminator
										// place 64-bit input data bit count
	*(unsigned *) &pad[padc - 8] = pmd5->totbyt[0] << 3;
	*(unsigned *) &pad[padc - 4] =
		(pmd5->totbyt[1] << 3) + (pmd5->totbyt[0] >> 32 - 3);
	Phmd5Process(pmd5, pad, padc);		// process the pad
}

// Work done here -- call for as many input blocks that need to be processed.
// pdata points to the input data, bytecnt is pdata size (0..n bytes).
// See phmd5.h regarding how to use this optimally.
void Phmd5Process(PHMD5 *pmd5, char *pdata, unsigned bytecnt) {
	unsigned resid = pmd5->totbyt[0];

	pmd5->totbyt[0] += bytecnt;			// update total bytes processed
	pmd5->totbyt[1] += pmd5->totbyt[0] < resid;

	resid &= 63;						// count of bytes now in pmd5->buf

	// The below "if" block could be removed and everything would still work.
	// What the if block does is process input data in-place, if the caller
	// continually provides it to us dword aligned and in 64-byte chunks.

	if (resid == 0 && bytecnt & ~63 && ((unsigned) pdata & 3) == 0) {
		Phmd5DoBlocks(pmd5->hash, pdata, bytecnt & ~63);
		pdata += bytecnt & ~63;
		bytecnt &= 63;
	}

	while (bytecnt) {
		unsigned cb = 64 - resid;
		if (bytecnt < cb) cb = bytecnt;
		memcpy(pmd5->buf + resid, pdata, cb);
		pdata += cb;
		bytecnt -= cb;
		resid = (resid + cb) & 63;
		if (resid) break;
		Phmd5DoBlocks(pmd5->hash, pmd5->buf, 64);
	};
}
