/*----------------------------------------------------------------------------
;
; phsrec.c	A simple Win32 command line raw wave audio capture application.
;
;			Creates PCM (.wav) output files recorded from a wave input device.
;			Uses the low-level (waveform audio) API.
;			No modification (compression) of the data is attempted.
;
;			Uses large buffering and event-driven audio capture logic to
;			avoid sample loss; also, disk output file is pre-allocated to
;			optimize writes while capturing, and disk writes are insured
;			to be in multiples of complete sectors.
;
;			compile with:	cl phsrec.c winmm.lib
;							upx phsrec.exe  (to shrink executable)
;
;  Date	 Programmer	Description
; ------ ----------	-----------
; 010614 P.Houle	Initial version.
; 081019 p.Houle	Added signal level to progress display.
;
;---------------------------------------------------------------------------*/

#pragma warning(disable:4201;disable:4214;disable:4514;disable:4115)
#pragma warning(disable:4032)
#include <stdio.h>
#include <stdlib.h>
#include <io.h>
#include <conio.h>
#include <fcntl.h>
#include <windows.h>
#include <sys\stat.h>
#pragma warning(default:4201;default:4214;default:4115;default:4032)


char fname0[_MAX_PATH] = {'\0'};		// command-line 1st file name
FILE *fp0;								// fname0 file handle
char *pfwb;								// ptr to file write buffer (or NULL)
long fwsize = 65536l * 2;				// size of buffer allocated at *pfwb
long fwbytes;							// data buffered (at front of buffer)
static void FlushBuffer(void) {			// flushes all data in write buffer
	if (fwbytes == 0) return;
	if ((long) fwrite(pfwb, 1, fwbytes, fp0) != fwbytes) { // n-sector write
		fprintf(stderr, "Error writing to file: %s\n", fname0);
		exit(-1);
	}
	fwbytes = 0;
}

int prompt = 0;							// non-zero to prompt before starting
int quiet = 0;							// non-zero to stop progress display
unsigned long LevAcc[3] = {0, 0, 0};

static void GetArgs(int argc, char **argv);
static void InitOutfile(void);
static void UpdLevAcc(char *pSamp, unsigned nbytes);

BOOL conAbrt = FALSE;	// set to true if console abort sensed
BOOL WINAPI CtrlHandler(DWORD Ctl) {	// Ctrl/Brk handler
	switch (Ctl) {
	  case CTRL_C_EVENT:
	  case CTRL_BREAK_EVENT:
		return conAbrt = TRUE;
	  default:
		return FALSE;
	}
}

typedef struct {						// used to control capture process
	int csecs;							// seconds to capture
	int BufSecs;						// I/O buff size, in secs (must be >0)
	int BufPerSec;						// # buffers per second of data
	int InDevId;						// input device +1 (0 for WAVE_MAPPER)
	WAVEFORMATEX wex;					// I/O: channels, samp/sec, bits/samp
} CAPCTL;
CAPCTL cc = {
  /*csecs*/5, /*BufSecs*/5, /*BufPerSec*/10, /*InDevId*/0,
  {0, /*channels*/2, /*freq*/44100, 0, 0, /*bits/sample*/16}
};

void CaptureAudio();

void main(int argc, char **argv) {

	GetArgs(argc, argv);				// get command line args, init fields
	InitOutfile();						// init/preallocate output file

	SetConsoleCtrlHandler(CtrlHandler, TRUE); // intercept & handle Ctrl-C/Brk

	CaptureAudio();						// capture audio data to disk

	FlushBuffer();						// flush any residual data
	free(pfwb);							// and free the disk write buffer

	exit(0);
}


/*----------------------------------------------------------------------------
;
; PrintInstructions() -- Prints usage info.
;
;---------------------------------------------------------------------------*/

static void PrintInstructions() {
  printf(
	"\n"
	"Audio capture tool, ver 1.1.  Paul Houle (paulhoule.com) Oct 19, 2008.\n"
	"Efficient capture (no dropped data) of raw audio to .wav file.\n\n"
	"Usage:  phsrec file{.wav} {/opts}\n\n"
	"  file.wav: Output wave file (\"-\" for stdout) - rewritten in-place\n"
	"            if already exists to avoid allocation time.\n"
	"{/opts}:\n"
	"  -?        This message.\n"
	"  -b{8|16}  Bits/Sample: 8 or 16 (default %d).\n"
	"  -c{1|2}   Number of channels: 1= mono, 2= stereo (default %d)\n"
	"  -f{nnn}   Capture samples/sec: e.g., 22050 (default %d)\n"
	"  -s{nn}    Number of seconds to capture (default %d).\n"
	"  -i{n}     Input wave device Id +1, 0 for WAVE_MAPPER (default %d).\n"
	"  -p        Prompt for keystroke before beginning recording.\n"
	"  -q        Quiet mode (no progress display)\n"
	"  -#q{nn}   Audio buffer queue size in seconds, 1..60 (default %d).\n"
	"  -#b{nnn}  Wave buffers/sec, 1..20 (default %d).\n"
	"  -#w{nnnn} Size of each disk write (minimum is 512, default is %d).\n"
	,cc.wex.wBitsPerSample ,cc.wex.nChannels ,cc.wex.nSamplesPerSec ,cc.csecs
	,cc.InDevId ,cc.BufSecs ,cc.BufPerSec ,fwsize
  );
  exit(-1);
}

/*----------------------------------------------------------------------------
; GetArgs() -- Process input command line.
;---------------------------------------------------------------------------*/

static void GetArgs(int argc, char **argv) {
	int iarg;

	for (iarg = 1; iarg < argc; iarg++) {
		if ((argv[iarg][0] == '-' || argv[iarg][0] == '/') && argv[iarg][1]) {
			char copt = argv[iarg][1];
			int a2i = atoi(argv[iarg] + 2);	// (arg as an integer)
			if (copt >= 'a' && copt <= 'z') copt -= 'a' - 'A';
			if (copt == '?') {
				PrintInstructions();
			} else if (copt == 'S') {
				if (a2i <= 0) break;
				cc.csecs = a2i;
			} else if (copt == 'C') {
				if (a2i != 1 && a2i != 2) break;
				cc.wex.nChannels = (WORD) a2i;
			} else if (copt == 'B') {
				if (a2i != 8 && a2i != 16) break;
				cc.wex.wBitsPerSample = (WORD) a2i;
			} else if (copt == 'F') {
				cc.wex.nSamplesPerSec = (WORD) a2i;
			} else if (copt == 'I') {
				cc.InDevId = a2i;
			} else if (copt == 'P') {
				prompt = 1;
			} else if (copt == 'Q') {
				quiet = 1;

			} else if (copt == '#') {
				copt = argv[iarg][2];
				if (copt >= 'a' && copt <= 'z') copt -= 'a' - 'A';
				a2i = atoi(argv[iarg] + 3);
				if (copt == 'Q') {
					if (a2i < 1 || a2i > 60) break;
					cc.BufSecs = a2i;
				} else if (copt == 'B') {
					if (a2i < 1 || a2i > 20) break;
					cc.BufPerSec = a2i;
				} else if (copt == 'W') {
					if (a2i < 512) break;
					fwsize = a2i & ~(512 - 1);	// (write only full sectors)
				} else {
					break;
				}

			} else {
				break;					// abort (illegal switch)
			}
		} else {
			if (fname0[0]) {
				fprintf(stderr, "Too many file names. Get help by\n"
					"starting the program without arguments.\n");
				exit(-1);
			}
			fname0[_MAX_PATH - 1] = '\0';
			strncpy(fname0, argv[iarg], _MAX_PATH - 1);
		}
	}

	if (iarg < argc) {					// if switch error,
		fprintf(stderr, "Illegal switch '%s', or bad switch value.\n"
			"Get help by starting the program without arguments.\n",
			argv[iarg]);
		exit(-1);
	}
	if (fname0[0] == '\0') PrintInstructions();

	if (cc.BufSecs > cc.csecs) {	// limit buffer secs to tot secs captured
		cc.BufSecs = cc.csecs;
	}

		// now that we have user opts, we can finish the wex. structure

	cc.wex.wFormatTag = WAVE_FORMAT_PCM;	// we only capture PCM
	cc.wex.nBlockAlign = (WORD)				// bytes per sample
		((cc.wex.nChannels * cc.wex.wBitsPerSample) / 8);
	cc.wex.nAvgBytesPerSec = cc.wex.nSamplesPerSec * cc.wex.nBlockAlign;
	cc.wex.cbSize = 0;
}

/*----------------------------------------------------------------------------
; AddOutput() -- Writes data to output buffer (flushing if buffer fills).
;---------------------------------------------------------------------------*/

void AddOutput(char *pbuf, long nbytes) {	// add data to output buffer
	for (;;) {
		long avail = fwsize - fwbytes;	// freespace in output buffer
		if (avail > nbytes) avail = nbytes;	// limit to amount we have to add
		memcpy(pfwb + fwbytes, pbuf, avail);	// add new data to buffer
		fwbytes += avail;
		if (nbytes == avail) return;	// return if all data fit in buffer
		nbytes -= avail;				// advance over data that did fit
		pbuf += avail;
		FlushBuffer();					// flush full buffer to disk
	}
}

/*----------------------------------------------------------------------------
; GenerateHeader() -- Generates RIFF header into the output buffer.
; Note the data size of the result file is in completed BUFFERS, not
; seconds -- this is to support exception cases.
; Returns the size in bytes of the entire RIFF file.
;---------------------------------------------------------------------------*/
long GenerateHeader(long bufs
) {
	typedef struct {				// RIFF hdr at start of .wav file
		char RIFF[4];
		DWORD fsize;
		char FormType[4];
		char ChunkID[4];
		DWORD hwexSize;
		char hwex[sizeof(WAVEFORMATEX) - 2]; // (less trailing .cbSize field)
		char DataID[4];
		DWORD DataSize;
	} RHDR;
	static RHDR rhdr = {
		{'R','I','F','F'}, 0, {'W','A','V','E'}, {'f','m','t',' '},
		sizeof(WAVEFORMATEX), {0}, {'d','a','t','a'}
	};
	UINT bsiz =							// size of normal buffer
		(cc.wex.nSamplesPerSec / cc.BufPerSec) * cc.wex.nBlockAlign;

	// Calc values of and fill out remaining rhdr. fields.

	rhdr.hwexSize = sizeof(rhdr.hwex);
	memcpy(&rhdr.hwex[0], &cc.wex, sizeof(rhdr.hwex));

	rhdr.DataSize =
		(DWORD) (bufs / cc.BufPerSec) * cc.wex.nAvgBytesPerSec
		+ (DWORD) (bufs % cc.BufPerSec) * bsiz;
	rhdr.fsize =						// (struct size is w/o RIFF hdr)
		(sizeof(rhdr) + rhdr.DataSize) - 8;
	
	AddOutput((char *)&rhdr, sizeof(rhdr));	// buffer the created RIFF hdr


	return rhdr.fsize + 8;				// return full size of output file
}

/*----------------------------------------------------------------------------
; InitOutfile() -- Init/preallocate output file.  We preallocate the file to
; remove the overhead associated with allocation while recording is underway.
;---------------------------------------------------------------------------*/

static void InitOutfile(void) {
	static char fdrive[_MAX_DRIVE], fdir[_MAX_DIR], fname[_MAX_FNAME];
	static char fext[_MAX_EXT];
	unsigned long fpsize;			// size in bytes of output file
	unsigned long exsize;			// size in bytes of file we're replacing
	struct _stat sb;				// file statistics buffer


	fwbytes = 0;					// allocate/empty our write buffer
	pfwb = (char *) malloc(fwsize);
	if (pfwb == NULL) {
		fprintf(stderr, "Couldn't allocate disk output buffer\n");
		exit(-1);
	}

	fpsize = GenerateHeader(			// generate output file RIFF header
		cc.csecs * cc.BufPerSec);

	// Fix up name & open output file,

	exsize = 0;							// indicate no re-use of existing file
	if (fname0[0] == '-') {				// if asking for stdout,
		fp0 = stdout;					// pipe output
		setmode(fileno(fp0), O_BINARY);	// make sure we're in binary mode
	} else {
		_splitpath(fname0, fdrive, fdir, fname, fext);
		if (fext[0] == '\0') strcpy(fext, ".wav");
		_makepath(fname0, fdrive, fdir, fname, fext);
		if (_stat(fname0, &sb) == 0) {	// try to get existing file size
			exsize = (unsigned long) sb.st_size;
		}

		if (exsize != 0) {				// if file exists (w/non-zero size),
			fp0 = fopen(fname0, "r+b");	// we'll write over in in-place
		} else {						// else none existing, create file,
			fp0 = fopen(fname0, "wb");
		}
		if (fp0 == NULL) {
			fprintf(stderr, "Error opening output file: %s\n", fname0);
			exit(-1);
		}
	}

	// Preallocate file -- leave positioned at start to write header
	// Note: we don't pre-allocate if piping to stdout ("-" output filename)

	for (;;) {							// lots of idiot checks below
		if (fname0[0] != '-') {			// preallocate only if not stdout
			if (fpsize != exsize) {		// if size needs to change,
				if (fpsize > exsize) {	// if it needs to grow,
						// this is faster than chsize() -- just allocates,
						// doesn't zero newly allocated sectors...
					if (fseek(fp0, fpsize - 1, SEEK_SET)) break;
					if (fwrite(&fwbytes, 1, 1, fp0) != 1) break;
					if (ftell(fp0) != (long) fpsize) break;
				} else {				// else it needs to shrink,
					if (chsize(fileno(fp0), fpsize)) break;
				}
				fclose(fp0);			// update file's directory size
				fp0 = fopen(fname0, "r+b");	// re-open file
				if (fp0 == NULL) break;
			}
			if (fseek(fp0, 0, SEEK_SET)) break;
		}
		fpsize = 0;					// this indicates success
		break;
	}
	if (fpsize != 0) {
		if (fp0 != NULL) fclose(fp0);	// close file, if open
		fprintf(stderr, "Error initialzing output: %s\n", fname0);
		fprintf(stderr, "Could be bad drive, bad file name,"
			" or disk full.\n");
		exit(-1);
	}
}

/*----------------------------------------------------------------------------
; Capture Logic.
;---------------------------------------------------------------------------*/

typedef struct {						// WAVEHDR FIFO Queue
	WAVEHDR **Head;						// head (oldest) queue entry
	WAVEHDR **Tail;						// tail (newest) queue entry
	WAVEHDR **Start;					// 1st entry slot ptr (queue start)
	WAVEHDR **End;						// points 1 past last entry slot
} WHFQ;
#define WHFQ_APPEND(Q, pWh) {			/* appends wavehdr ptr */			\
	WAVEHDR **Tail = Q.Tail;			/* load queue tail (write) ptr */	\
	*Tail++ = (WAVEHDR *) (pWh);		/* append to queue/advance tail */	\
	if (Tail == Q.End) Tail = Q.Start;	/* wrap queue ptr if needed */		\
	Q.Tail = Tail;						/* update ptr in memory */			\
}
#define WHFQ_EXTRACT(Q, pWh) {			/* reads/removes wave hdr ptr */	\
	WAVEHDR **Head = Q.Head;			/* load queue head (read) ptr */	\
	pWh = *Head++;						/* read entry/advance head */		\
	if (Head == Q.End) Head = Q.Start;	/* wrap queue ptr if needed */		\
	Q.Head = Head;						/* update ptr in memory */			\
}

typedef struct {						// WAVEHDR control & buffer mgt
	HANDLE hev;							// event used for I/O completion wait
	HWAVEIN hwi;						// handle to wave input device
	DWORD totbufs;						// total buffers of data we'll capture
	DWORD addbufs;						// buffers "added" to input queue
	DWORD donebufs;						// count of total "done" buffers

	WHFQ RecQ;							// "done" record header queue

	WAVEHDR wh[1];						// 1st wave header (of alloc'd array)
} WHCTL;


void CALLBACK waveInProc(				// wave in I/O completion procedure
	HWAVEIN hwi, UINT uMsg, DWORD dwInstance, DWORD dwParam1, DWORD dwParam2
) {
	hwi = hwi; dwParam2 = dwParam2;		// (silence compiler warnings)
	if (uMsg == WIM_DATA) {				// only respond to buffer "done" msgs,
		WHCTL *pwc = (WHCTL *) dwInstance; // instance value is WHCTL ptr
		WHFQ_APPEND(pwc->RecQ, dwParam1) // append to rec done queue
		SetEvent(pwc->hev);				// indicate I/O completion occurred
	}
}

void wiErrorFmt(char *fn, MMRESULT ires
) {
	static char ebuf[MAXERRORLENGTH];
	if (ires == 0) return;
	ebuf[0] = '\0';
	waveInGetErrorText(					// get input error description string
		ires, &ebuf[0], sizeof(ebuf));
	fprintf(stderr, "*** Capture error during %s: %s\n", fn, ebuf);
}

void UpdDisp(DWORD dbuf, DWORD tbuf
) {
	static DWORD lastdb = 0;
	unsigned long lvt;

	/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Don't display if quiet mode OR donebufs == lastdb  -- inhibits display
	; when donebufs is zero, or when it matches what's currently onscreen.
	; (No display at zero since there's no data to calc volume levels from.)
	;
	; Do display if exactly 1 "done" buffer OR on a 1-second boundary.
	*/

	if (quiet || dbuf == lastdb) return;
	lvt = dbuf % cc.BufPerSec;
	if (!(dbuf == 1 || lvt == 0)) return;

	lastdb = dbuf;
	dbuf = (tbuf - (dbuf - lvt)) / cc.BufPerSec;
	lvt = max(LevAcc[2], 1);
	fprintf(stderr, "\r%3d:%02d:%02d  Avg Level:%4u",
		dbuf / 3600, (dbuf / 60) % 60, dbuf % 60, LevAcc[0] / lvt);
	if (cc.wex.nChannels == 2) fprintf(stderr, "%4u", LevAcc[1] / lvt);
	memset(LevAcc, 0, sizeof(LevAcc));
}

void CaptureAudio(
) {
	MMRESULT ires = 0;					// input result code
	WHCTL *pwc = (WHCTL *) NULL;		// wave-device related fields
	WAVEHDR *pwh;						// ptr to wave header

  for (;;) {

	/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; We allocate the wave-related control fields, handles, and objects, and
	; all associated queues and buffers, as one large block addressed by pwc.
	; The sections of the block are (in order):
	;
	;	(1)	The leading control field structure (WHCTL).
	;
	;	(2)	The array of all waveheaders.  This immediately follows WHCTL
	;		in memory; also, the last field of WHCTL (.wh[]) is used as a
	;		placeholder to provide direct access (via pwc->) to the
	;		start of the dynamically sized wave header array.
	;
	;	(3)	The entries for the record "done" block queue.  There are enough
	;		free slots to hold all possible headers, so an out-of-room
	;		condition is never possible.
	;
	;	(4)	The data buffer(s) associated with each wave header.
	*/

	{
		UINT nhdr =						// number of wave hdrs/buffers
			cc.BufSecs * cc.BufPerSec;
		UINT csiz =						// size of (leading) control area
			(sizeof(WHCTL) - sizeof(WAVEHDR)) + nhdr * sizeof(WAVEHDR);
		UINT bsiz =						// size of normal buffer
			(cc.wex.nSamplesPerSec / cc.BufPerSec) * cc.wex.nBlockAlign;
		UINT bres =						// residual (in last buffer each sec)
			(cc.wex.nSamplesPerSec % cc.BufPerSec) * cc.wex.nBlockAlign;
		UINT i;							// temp

		pwc = (WHCTL *) malloc(			// alloc: leading control area
			csiz
		  + (nhdr + 1) * sizeof(WAVEHDR *)		// rec queue
		  + cc.BufSecs * cc.wex.nAvgBytesPerSec	// wave hdr data buffers
		);
		ires = MMSYSERR_NOMEM;			// assume memory error
		if (!pwc) break;				// abort= fatal error
		memset(pwc, 0, csiz);			// init all control fields to zero

		pwc->totbufs = cc.csecs * cc.BufPerSec; // init total bufs to capture

		pwc->RecQ.Start =				// init record "done" queue
			pwc->RecQ.Head =
			pwc->RecQ.Tail = (WAVEHDR **) (csiz + (char *) pwc);
		pwc->RecQ.End = pwc->RecQ.Start + (nhdr + 1);

		csiz = (char *) (pwc->RecQ.End) - (char *) pwc;

		for (i = 0; i < nhdr; i++) {	// for all headers,
			pwc->wh[i].lpData =			// set data buffer address
				(LPSTR) (csiz + (char *) pwc);
			pwc->wh[i].dwBufferLength =	// set data buffer size in bytes
				bsiz + (i % cc.BufPerSec == 0 ? bres : 0);
			csiz += pwc->wh[i].dwBufferLength; // advance next buff startpoint
		}
	}

	pwc->hev = CreateEvent(				// create OS event object
		NULL,							// default security attributes
		TRUE,							// event must be manually reset
		FALSE,							// initial state = nonsignaled
		NULL							// no name for this event
	);
	if (!pwc->hev) {					// if failure creating event object,
		ires = MMSYSERR_ERROR;			// generate non-specific error & abort
		break;
	}

	ires = waveInOpen(					// open wave input device
		&pwc->hwi,						// handle to create
		(UINT) (cc.InDevId ?			// wave input device ID
			cc.InDevId - 1 : WAVE_MAPPER),
		&cc.wex,						// PCM format (freq, channels, etc)
		(DWORD) waveInProc,				// callback fn we'll use
		(DWORD) pwc,					// callback instance data
			CALLBACK_FUNCTION			//	use callback func for completion
	);
	if (ires) {							// if open failed,
		pwc->hwi = NULL;				// insure handle is NULL
		break;							// and abort
	}

	/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Prepare and add all the record buffers to the wave system input queue.
	; Note we don't worry about adding too many buffers here -- we insured
	; earlier (during GetArgs) that cc.BufSecs <= cc.secs.
	*/

	{
		UINT nhdr =						// number of wave hdrs/buffers
			cc.BufSecs * cc.BufPerSec;
		UINT i;							// temp

		for (i = 0; i < nhdr; i++) {	// for all headers,
			ires = waveInPrepareHeader(	// "prepare" the buffer for input
				pwc->hwi, &pwc->wh[i], sizeof(pwc->wh[i]));
			if (ires) break;			// break= error preparing

			ires = waveInAddBuffer(		// add buffer to input queue
				pwc->hwi, &pwc->wh[i], sizeof(pwc->wh[i]));
			if (ires) break;			// break= error adding buffer
			++pwc->addbufs;				// indicate another buffer was "added"
		}
	}
	if (ires) break;					// break= fatal error, abort

	if (prompt) {						// if prompting before record start,
		fprintf(stderr, "Ready to record -- press any key to begin...");
		getch();
		fprintf(stderr, "\n");
	}

	ires = waveInStart(pwc->hwi);		// start capture
	if (ires) break;					// break= fatal error, abort


	while (!conAbrt) {					// while no Ctrl-C or Ctrl-Brk,

		UpdDisp(pwc->donebufs, pwc->totbufs);

		if (pwc->RecQ.Head == pwc->RecQ.Tail) { // if no "done' record buffer,
			if (WaitForSingleObject(	// wait for event signal (done buffer)
			  pwc->hev, 2500) == WAIT_OBJECT_0) {	// if signaled OK,
				if (ResetEvent(pwc->hev)) continue; // reset event & continue
			}
			ires = MMSYSERR_ERROR;		// abort, unexpected error
			break;
		}

		/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		; It is wise to "unprep/prep" to re-use an input buffer before
		; re-adding -- read this app-note from Dr. Dobbs magazine:
		;
		;	Although Microsoft's documentation does not say that
		;	unpreparing a recycled header is required, some Audio
		;	Compression Manager (ACM) drivers certainly appear to make
		;	that assumption.  Without an additional unprepare call, some
		;	ACM drivers leak memory and/or act improperly.  If you are
		;	directly recording formats other than PCM, the sequence I
		;	would use to recycle a buffer is:
		;
		;		waveInUnprepareHeader(hdr, ...);
		;		waveInPrepareHeader(hdr, ...);
		;		waveInAddBuffer(hdr, ...);
		;
		;	Even if you aren't explicitly asking for a nonPCM format, it
		;	pays to be careful because the ACM manager can insert ACM
		;	drivers into the data stream if necessary to supply formats
		;	that your hardware will not directly support.  Probably the
		;	best example of this would be a sample rate converter that
		;	might get inserted if your card supported only 22 KHz, but
		;	you asked for 44 KHz.
		*/

		WHFQ_EXTRACT(pwc->RecQ, pwh)	// extract from rec done queue

		ires = waveInUnprepareHeader(	// "unprepare" filled buffer
			pwc->hwi, pwh, sizeof(*pwh));
		if (ires) break;				// abort= fatal error

		if (pwh->dwBytesRecorded != pwh->dwBufferLength) {
			ires = MMSYSERR_ERROR;		// abort -- partially filled buffer
			break;
		}
		if (!quiet) {
			UpdLevAcc(pwh->lpData, pwh->dwBytesRecorded);
		}

		AddOutput(pwh->lpData, pwh->dwBytesRecorded); // output to disk
		pwh->dwBytesRecorded = 0;		// show buffer is now empty
										// break if all data captured
		if (++pwc->donebufs == pwc->totbufs) break;

		if (pwc->addbufs < pwc->totbufs) {	// if more buffers left to add,
			pwh->dwFlags = 0;			// (must zero prior to preparing)
			ires = waveInPrepareHeader(	// prepare buffer (again) for input
				pwc->hwi, pwh, sizeof(*pwh));
			if (ires) break;			// break= fatal error, abort

			ires = waveInAddBuffer(		// add buffer back into input queue
				pwc->hwi, pwh, sizeof(*pwh));
			if (ires) break;			// break= fatal error, abort
			++pwc->addbufs;				// bump count of added buffers
		}
	}

	break;
  }

	wiErrorFmt("unknown", ires);		// display any pending error msg

	UpdDisp(pwc->totbufs, pwc->totbufs); // finalize progress display

	/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; If there's been some kind of exception (Ctrl-c, Ctrl-Brk, or a wave
	; error), we need to reset to insure any pending added buffers are
	; immediately released back to us (i.e., marked "done").  Those buffers
	; also need to be unprepared -- to handle that, we simply unprepare all
	; the buffers we created (it's harmless to unprepare a buffer that is not
	; in the prepared state).
	*/

	if ((conAbrt || ires)				// if some kind of exception occurred,
	 && pwc && pwc->hwi) {				// and Wave input is open,
		int i;
		wiErrorFmt("waveInStop (err)",	// stop input
			waveInStop(pwc->hwi));
		wiErrorFmt("waveInReset (err)", // mark all bufs done
			waveInReset(pwc->hwi));
			// Note: on rare occasion the unprepare will generate an error
			// msg (buffer still in queue).  That shouldn't be possible due
			// to the above reset, but it does happen RARELY.  An error msg
			// pops up, but no fatal side-effects seem to happen.
		for (i = 0; i < cc.BufSecs * cc.BufPerSec; i++) { // for all headers,
		  wiErrorFmt("waveInUnprepareHeader (err)", // "unprepare" all buffers
			waveInUnprepareHeader(pwc->hwi, &pwc->wh[i], sizeof(pwc->wh[i])));
		}
		if (conAbrt) fprintf(stderr,
			"*** Keyboard abort -- truncating output...\n");
		FlushBuffer();					// flush out any done data
		if (fseek(fp0, 0, SEEK_SET) == 0) {	// if reposition to start works,
			long fpsize = GenerateHeader( // generate shortened file RIFF hdr
				pwc->donebufs);
			FlushBuffer();				// update RIFF hdr to shorter size
			chsize(fileno(fp0), fpsize); // truncate file to new length
		}
	}

	/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Shutdown recording, release buffers.
	*/

	if (pwc) {							// if main instance block allocated,
		if (pwc->hwi) {					// if wave input is open,
			wiErrorFmt("waveInReset",	// stop input, mark bufs done
				waveInReset(pwc->hwi));
			wiErrorFmt("waveInStop",	// prob superfluous now...
				waveInStop(pwc->hwi));
			wiErrorFmt("waveInClose",	// close wave input device
				waveInClose(pwc->hwi));
		}

		if (pwc->hev) {					// if event object exists,
			CloseHandle(pwc->hev);		// release it
		}

		free(pwc);						// free the main block
	}
}

/*----------------------------------------------------------------------------
; Signal Level Accumulator.		(added October 18, 2008)
;---------------------------------------------------------------------------*/

static void UpdLevAcc(char *pSamp, unsigned nbytes) {
	unsigned long c0Acc = 0;
	unsigned long c1Acc = 0;
	char *pEnd = pSamp + nbytes;		// end address

	if (cc.wex.wBitsPerSample == 8) {	// if 8-bit cases
		if (cc.wex.nChannels == 1) {	// if mono,
			while (pSamp < pEnd) {
				c0Acc += abs(((unsigned char *)pSamp)[0] - 0x80);
				pSamp += 1;
			}
		} else {						// else stereo,
			while (pSamp < pEnd) {
				c0Acc += abs(((unsigned char *)pSamp)[0] - 0x80);
				c1Acc += abs(((unsigned char *)pSamp)[1] - 0x80);
				pSamp += 2;
			}
		}
	} else {							// else 16-bit cases,
		if (cc.wex.nChannels == 1) {	// if mono,
			while (pSamp < pEnd) {
				c0Acc += abs(((short *)pSamp)[0]) >> 8;
				pSamp += 2;
			}
		} else {						// else stereo,
			while (pSamp < pEnd) {
				c0Acc += abs(((short *)pSamp)[0]) >> 8;
				c1Acc += abs(((short *)pSamp)[1]) >> 8;
				pSamp += 4;
			}
		}
	}
	LevAcc[0] += c0Acc;					// update memory accumulators
	LevAcc[1] += c1Acc;
	LevAcc[2] +=						// update samples accumulated
		nbytes / (cc.wex.nChannels * (cc.wex.wBitsPerSample / 8));
}
